# -*- coding: utf-8 -*-
# Copyright (c) 2022, Felix Fontein (@felixfontein) <felix@fontein.de>
# GNU General Public License v3.0+ (see LICENSES/GPL-3.0-or-later.txt or https://www.gnu.org/licenses/gpl-3.0.txt)
# SPDX-License-Identifier: GPL-3.0-or-later

# The data inside here is private to this collection. If you use this from outside the collection,
# you are on your own. There can be random changes to its format even in bugfix releases!

from __future__ import absolute_import, division, print_function
__metaclass__ = type

from ansible_collections.community.routeros.plugins.module_utils.version import LooseVersion


def _compare(a, b, comparator):
    if comparator == '==':
        return a == b
    if comparator == '!=':
        return a != b
    if comparator == '<':
        return a < b
    if comparator == '<=':
        return a <= b
    if comparator == '>':
        return a > b
    if comparator == '>=':
        return a >= b
    raise ValueError('Unknown comparator "{comparator}"'.format(comparator=comparator))


class APIData(object):
    def __init__(self,
                 unversioned=None,
                 versioned=None):
        if (unversioned is None) == (versioned is None):
            raise ValueError('either unversioned or versioned must be provided')
        self.unversioned = unversioned
        self.versioned = versioned
        if self.unversioned is not None:
            self.needs_version = self.unversioned.needs_version
            self.fully_understood = self.unversioned.fully_understood
        else:
            self.needs_version = self.versioned is not None
            # Mark as 'fully understood' if it is for at least one version
            self.fully_understood = False
            for dummy, dummy, unversioned in self.versioned:
                if unversioned.fully_understood:
                    self.fully_understood = True
                    break
        self._current = None if self.needs_version else self.unversioned

    def provide_version(self, version):
        if not self.needs_version:
            return self.unversioned.fully_understood
        api_version = LooseVersion(version)
        if self.unversioned is not None:
            self._current = self.unversioned.specialize_for_version(api_version)
            return self._current.fully_understood
        for other_version, comparator, data in self.versioned:
            if other_version == '*' and comparator == '*':
                self._current = data.specialize_for_version(api_version)
                return self._current.fully_understood
            other_api_version = LooseVersion(other_version)
            if _compare(api_version, other_api_version, comparator):
                self._current = data.specialize_for_version(api_version)
                return self._current.fully_understood
        self._current = None
        return False

    def get_data(self):
        if self._current is None:
            raise ValueError('either provide_version() was not called or it returned False')
        return self._current


class VersionedAPIData(object):
    def __init__(self,
                 primary_keys=None,
                 stratify_keys=None,
                 required_one_of=None,
                 mutually_exclusive=None,
                 has_identifier=False,
                 single_value=False,
                 unknown_mechanism=False,
                 fully_understood=False,
                 fixed_entries=False,
                 fields=None,
                 versioned_fields=None):
        if sum([primary_keys is not None, stratify_keys is not None, has_identifier, single_value, unknown_mechanism]) > 1:
            raise ValueError('primary_keys, stratify_keys, has_identifier, single_value, and unknown_mechanism are mutually exclusive')
        if unknown_mechanism and fully_understood:
            raise ValueError('unknown_mechanism and fully_understood cannot be combined')
        self.primary_keys = primary_keys
        self.stratify_keys = stratify_keys
        self.required_one_of = required_one_of or []
        self.mutually_exclusive = mutually_exclusive or []
        self.has_identifier = has_identifier
        self.single_value = single_value
        self.unknown_mechanism = unknown_mechanism
        self.fully_understood = fully_understood
        self.fixed_entries = fixed_entries
        if fixed_entries and primary_keys is None:
            raise ValueError('fixed_entries can only be used with primary_keys')
        if fields is None:
            raise ValueError('fields must be provided')
        self.fields = fields
        if versioned_fields is not None:
            if not isinstance(versioned_fields, list):
                raise ValueError('unversioned_fields must be a list')
            for conditions, name, field in versioned_fields:
                if not isinstance(conditions, (tuple, list)):
                    raise ValueError('conditions must be a list or tuple')
                if not isinstance(field, KeyInfo):
                    raise ValueError('field must be a KeyInfo object')
                if name in fields:
                    raise ValueError('"{name}" appears both in fields and versioned_fields'.format(name=name))
        self.versioned_fields = versioned_fields or []
        if primary_keys:
            for pk in primary_keys:
                if pk not in fields:
                    raise ValueError('Primary key {pk} must be in fields!'.format(pk=pk))
        if stratify_keys:
            for sk in stratify_keys:
                if sk not in fields:
                    raise ValueError('Stratify key {sk} must be in fields!'.format(sk=sk))
        if required_one_of:
            for index, require_list in enumerate(required_one_of):
                if not isinstance(require_list, list):
                    raise ValueError('Require one of element at index #{index} must be a list!'.format(index=index + 1))
                for rk in require_list:
                    if rk not in fields:
                        raise ValueError('Require one of key {rk} must be in fields!'.format(rk=rk))
        if mutually_exclusive:
            for index, exclusive_list in enumerate(mutually_exclusive):
                if not isinstance(exclusive_list, list):
                    raise ValueError('Mutually exclusive element at index #{index} must be a list!'.format(index=index + 1))
                for ek in exclusive_list:
                    if ek not in fields:
                        raise ValueError('Mutually exclusive key {ek} must be in fields!'.format(ek=ek))
        self.needs_version = len(self.versioned_fields) > 0

    def specialize_for_version(self, api_version):
        fields = self.fields.copy()
        for conditions, name, field in self.versioned_fields:
            matching = True
            for other_version, comparator in conditions:
                other_api_version = LooseVersion(other_version)
                if not _compare(api_version, other_api_version, comparator):
                    matching = False
                    break
            if matching:
                if name in fields:
                    raise ValueError(
                        'Internal error: field "{field}" already exists for {version}'.format(field=name, version=api_version)
                    )
                fields[name] = field
        return VersionedAPIData(
            primary_keys=self.primary_keys,
            stratify_keys=self.stratify_keys,
            required_one_of=self.required_one_of,
            mutually_exclusive=self.mutually_exclusive,
            has_identifier=self.has_identifier,
            single_value=self.single_value,
            unknown_mechanism=self.unknown_mechanism,
            fully_understood=self.fully_understood,
            fixed_entries=self.fixed_entries,
            fields=fields,
        )


class KeyInfo(object):
    def __init__(self,
                 _dummy=None,
                 can_disable=False,
                 remove_value=None,
                 absent_value=None,
                 default=None,
                 required=False,
                 automatically_computed_from=None,
                 read_only=False,
                 write_only=False):
        if _dummy is not None:
            raise ValueError('KeyInfo() does not have positional arguments')
        if sum([required, default is not None or can_disable, automatically_computed_from is not None]) > 1:
            raise ValueError(
                'required, default, automatically_computed_from, and can_disable are mutually exclusive '
                'besides default and can_disable which can be set together')
        if not can_disable and remove_value is not None:
            raise ValueError('remove_value can only be specified if can_disable=True')
        if absent_value is not None and any([default is not None, automatically_computed_from is not None, can_disable]):
            raise ValueError('absent_value can not be combined with default, automatically_computed_from, can_disable=True, or absent_value')
        if read_only and write_only:
            raise ValueError('read_only and write_only cannot be used at the same time')
        self.can_disable = can_disable
        self.remove_value = remove_value
        self.automatically_computed_from = automatically_computed_from
        self.default = default
        self.required = required
        self.absent_value = absent_value
        self.read_only = read_only
        self.write_only = write_only


def split_path(path):
    return path.split()


def join_path(path):
    return ' '.join(path)


# How to obtain this information:
# 1. Run `/export verbose` in the CLI;
# 2. All attributes listed there go into the `fields` list;
#    attributes which can have a `!` ahead should have `canDisable=True`
# 3. All bold attributes go into the `primary_keys` list -- this is not always true!

PATHS = {
    ('interface', 'bonding'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'arp': KeyInfo(default='enabled'),
                'arp-interval': KeyInfo(default='100ms'),
                'arp-ip-targets': KeyInfo(default=''),
                'arp-timeout': KeyInfo(default='auto'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'down-delay': KeyInfo(default='0ms'),
                'forced-mac-address': KeyInfo(can_disable=True),
                'lacp-rate': KeyInfo(default='30secs'),
                'lacp-user-key': KeyInfo(can_disable=True, remove_value=0),
                'link-monitoring': KeyInfo(default='mii'),
                'mii-interval': KeyInfo(default='100ms'),
                'min-links': KeyInfo(default=0),
                'mlag-id': KeyInfo(can_disable=True, remove_value=0),
                'mode': KeyInfo(default='balance-rr'),
                'mtu': KeyInfo(default=1500),
                'name': KeyInfo(),
                'primary': KeyInfo(default='none'),
                'slaves': KeyInfo(required=True),
                'transmit-hash-policy': KeyInfo(default='layer-2'),
                'up-delay': KeyInfo(default='0ms'),
            }
        ),
    ),
    ('interface', 'bridge'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'admin-mac': KeyInfo(default=''),
                'ageing-time': KeyInfo(default='5m'),
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'auto-mac': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dhcp-snooping': KeyInfo(default=False),
                'disabled': KeyInfo(default=False),
                'ether-type': KeyInfo(default='0x8100'),
                'fast-forward': KeyInfo(default=True),
                'frame-types': KeyInfo(default='admit-all'),
                'forward-delay': KeyInfo(default='15s'),
                'igmp-snooping': KeyInfo(default=False),
                'ingress-filtering': KeyInfo(default=True),
                'max-message-age': KeyInfo(default='20s'),
                'mtu': KeyInfo(default='auto'),
                'name': KeyInfo(),
                'priority': KeyInfo(default='0x8000'),
                'protocol-mode': KeyInfo(default='rstp'),
                'pvid': KeyInfo(default=1),
                'transmit-hold-count': KeyInfo(default=6),
                'vlan-filtering': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'eoip'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'allow-fast-path': KeyInfo(default=True),
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'clamp-tcp-mss': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'dont-fragment': KeyInfo(default=False),
                'dscp': KeyInfo(default='inherit'),
                'ipsec-secret': KeyInfo(can_disable=True),
                'keepalive': KeyInfo(default='10s,10', can_disable=True),
                'local-address': KeyInfo(default='0.0.0.0'),
                'loop-protect': KeyInfo(default='default'),
                'loop-protect-disable-time': KeyInfo(default='5m'),
                'loop-protect-send-interval': KeyInfo(default='5s'),
                'mac-address': KeyInfo(),
                'mtu': KeyInfo(default='auto'),
                'name': KeyInfo(),
                'remote-address': KeyInfo(required=True),
                'tunnel-id': KeyInfo(required=True),
            },
        ),
    ),
    ('interface', 'ethernet'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            fully_understood=True,
            primary_keys=('default-name', ),
            fields={
                'default-name': KeyInfo(),
                'advertise': KeyInfo(),
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'auto-negotiation': KeyInfo(default=True),
                'bandwidth': KeyInfo(default='unlimited/unlimited'),
                'combo-mode': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'fec-mode': KeyInfo(can_disable=True),
                'full-duplex': KeyInfo(default=True),
                'l2mtu': KeyInfo(default=1598),
                'loop-protect': KeyInfo(default='default'),
                'loop-protect-disable-time': KeyInfo(default='5m'),
                'loop-protect-send-interval': KeyInfo(default='5s'),
                'mac-address': KeyInfo(),
                'mdix-enable': KeyInfo(),
                'mtu': KeyInfo(default=1500),
                'name': KeyInfo(),
                'orig-mac-address': KeyInfo(),
                'poe-out': KeyInfo(can_disable=True),
                'poe-priority': KeyInfo(can_disable=True),
                'poe-voltage': KeyInfo(can_disable=True),
                'power-cycle-interval': KeyInfo(),
                'power-cycle-ping-address': KeyInfo(can_disable=True),
                'power-cycle-ping-enabled': KeyInfo(),
                'power-cycle-ping-timeout': KeyInfo(can_disable=True),
                'rx-flow-control': KeyInfo(default='off'),
                'sfp-rate-select': KeyInfo(default='high'),
                'sfp-shutdown-temperature': KeyInfo(default='95C'),
                'speed': KeyInfo(),
                'tx-flow-control': KeyInfo(default='off'),
            },
        ),
    ),
    ('interface', 'ethernet', 'poe'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'name': KeyInfo(),
                'poe-out': KeyInfo(default='auto-on'),
                'poe-priority': KeyInfo(default=10),
                'poe-voltage': KeyInfo(default='auto'),
                'power-cycle-interval': KeyInfo(default='none'),
                'power-cycle-ping-address': KeyInfo(can_disable=True),
                'power-cycle-ping-enabled': KeyInfo(default=False),
                'power-cycle-ping-timeout': KeyInfo(can_disable=True),
            }
        ),
    ),
    ('interface', 'gre'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'allow-fast-path': KeyInfo(default=True),
                'clamp-tcp-mss': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'dont-fragment': KeyInfo(default=False),
                'dscp': KeyInfo(default='inherit'),
                'ipsec-secret': KeyInfo(can_disable=True),
                'keepalive': KeyInfo(default='10s,10', can_disable=True),
                'local-address': KeyInfo(default='0.0.0.0'),
                'mtu': KeyInfo(default='auto'),
                'name': KeyInfo(),
                'remote-address': KeyInfo(required=True),
            },
        ),
    ),
    ('interface', 'gre6'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'clamp-tcp-mss': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(default='inherit'),
                'ipsec-secret': KeyInfo(can_disable=True),
                'keepalive': KeyInfo(default='10s,10', can_disable=True),
                'local-address': KeyInfo(default='::'),
                'mtu': KeyInfo(default='auto'),
                'name': KeyInfo(),
                'remote-address': KeyInfo(required=True),
            },
        ),
    ),
    ('interface', 'list'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'exclude': KeyInfo(),
                'include': KeyInfo(),
                'name': KeyInfo(),
            },
        ),
    ),
    ('interface', 'list', 'member'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('list', 'interface', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'interface': KeyInfo(),
                'list': KeyInfo(),
                'disabled': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'lte', 'apn'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'add-default-route': KeyInfo(),
                'apn': KeyInfo(),
                'default-route-distance': KeyInfo(),
                'name': KeyInfo(),
                'use-peer-dns': KeyInfo(),
            },
        ),
    ),
    ('interface', 'ppp-client'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'add-default-route': KeyInfo(default=True),
                'allow': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'apn': KeyInfo(default='internet'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'data-channel': KeyInfo(default=0),
                'default-route-distance': KeyInfo(default=1),
                'dial-command': KeyInfo(default="ATDT"),
                'dial-on-demand': KeyInfo(default=True),
                'disabled': KeyInfo(default=True),
                'info-channel': KeyInfo(default=0),
                'keepalive-timeout': KeyInfo(default=30),
                'max-mru': KeyInfo(default=1500),
                'max-mtu': KeyInfo(default=1500),
                'modem-init': KeyInfo(default=''),
                'mrru': KeyInfo(default='disabled'),
                'name': KeyInfo(),
                'null-modem': KeyInfo(default=False),
                'password': KeyInfo(default=''),
                'phone': KeyInfo(default=''),
                'pin': KeyInfo(default=''),
                'port': KeyInfo(),
                'profile': KeyInfo(default='default'),
                'use-peer-dns': KeyInfo(default=True),
                'user': KeyInfo(default=''),
            },
        ),
    ),
    ('interface', 'pppoe-client'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'ac-name': KeyInfo(default=''),
                'add-default-route': KeyInfo(default=False),
                'allow': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-route-distance': KeyInfo(default=1),
                'dial-on-demand': KeyInfo(default=False),
                'disabled': KeyInfo(default=True),
                'host-uniq': KeyInfo(can_disable=True),
                'interface': KeyInfo(required=True),
                'keepalive-timeout': KeyInfo(default=10),
                'max-mru': KeyInfo(default='auto'),
                'max-mtu': KeyInfo(default='auto'),
                'mrru': KeyInfo(default='disabled'),
                'name': KeyInfo(),
                'password': KeyInfo(default=''),
                'profile': KeyInfo(default='default'),
                'service-name': KeyInfo(default=''),
                'use-peer-dns': KeyInfo(default=False),
                'user': KeyInfo(default=''),
            },
        ),
    ),
    ('interface', 'vlan'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(required=True),
                'loop-protect': KeyInfo(default='default'),
                'loop-protect-disable-time': KeyInfo(default='5m'),
                'loop-protect-send-interval': KeyInfo(default='5s'),
                'mtu': KeyInfo(default=1500),
                'name': KeyInfo(),
                'use-service-tag': KeyInfo(default=False),
                'vlan-id': KeyInfo(required=True),
            },
        ),
    ),
    ('interface', 'vrrp'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'authentication': KeyInfo(default='none'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'group-master': KeyInfo(default=''),
                'interface': KeyInfo(required=True),
                'interval': KeyInfo(default='1s'),
                'mtu': KeyInfo(default=1500),
                'name': KeyInfo(),
                'on-backup': KeyInfo(default=''),
                'on-fail': KeyInfo(default=''),
                'on-master': KeyInfo(default=''),
                'password': KeyInfo(default=''),
                'preemption-mode': KeyInfo(default=True),
                'priority': KeyInfo(default=100),
                'remote-address': KeyInfo(),
                'sync-connection-tracking': KeyInfo(default=False),
                'v3-protocol': KeyInfo(default='ipv4'),
                'version': KeyInfo(default=3),
                'vrid': KeyInfo(default=1),
            },
        ),
    ),
    ('ip', 'hotspot', 'profile'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'dns-name': KeyInfo(),
                'hotspot-address': KeyInfo(),
                'html-directory': KeyInfo(),
                'html-directory-override': KeyInfo(),
                'http-cookie-lifetime': KeyInfo(),
                'http-proxy': KeyInfo(),
                'login-by': KeyInfo(),
                'name': KeyInfo(),
                'rate-limit': KeyInfo(),
                'smtp-server': KeyInfo(),
                'split-user-domain': KeyInfo(),
                'use-radius': KeyInfo(),
            },
        ),
    ),
    ('ip', 'hotspot', 'user', 'profile'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'add-mac-cookie': KeyInfo(),
                'address-list': KeyInfo(),
                'idle-timeout': KeyInfo(),
                'insert-queue-before': KeyInfo(can_disable=True),
                'keepalive-timeout': KeyInfo(),
                'mac-cookie-timeout': KeyInfo(),
                'name': KeyInfo(),
                'parent-queue': KeyInfo(can_disable=True),
                'queue-type': KeyInfo(can_disable=True),
                'shared-users': KeyInfo(),
                'status-autorefresh': KeyInfo(),
                'transparent-proxy': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'identity'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('peer', ),
            fields={
                'auth-method': KeyInfo(default='pre-shared-key'),
                'certificate': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'eap-methods': KeyInfo(default='eap-tls'),
                'generate-policy': KeyInfo(default=False),
                'key': KeyInfo(),
                'match-by': KeyInfo(can_disable=True, remove_value='remote-id'),
                'mode-config': KeyInfo(can_disable=True, remove_value='none'),
                'my-id': KeyInfo(can_disable=True, remove_value='auto'),
                'notrack-chain': KeyInfo(can_disable=True, remove_value=''),
                'password': KeyInfo(),
                'peer': KeyInfo(),
                'policy-template-group': KeyInfo(can_disable=True, remove_value='default'),
                'remote-certificate': KeyInfo(),
                'remote-id': KeyInfo(can_disable=True, remove_value='auto'),
                'remote-key': KeyInfo(),
                'secret': KeyInfo(default=''),
                'username': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'mode-config'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'name': KeyInfo(),
                'responder': KeyInfo(),
                'use-responder-dns': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'peer'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'address': KeyInfo(can_disable=True, remove_value=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'exchange-mode': KeyInfo(default='main'),
                'local-address': KeyInfo(can_disable=True, remove_value='0.0.0.0'),
                'name': KeyInfo(),
                'passive': KeyInfo(can_disable=True, remove_value=False),
                'port': KeyInfo(can_disable=True, remove_value=500),
                'profile': KeyInfo(default='default'),
                'send-initial-contact': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'ipsec', 'policy', 'group'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'name': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'profile'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'dh-group': KeyInfo(default='modp2048,modp1024'),
                'dpd-interval': KeyInfo(default='2m'),
                'dpd-maximum-failures': KeyInfo(default=5),
                'enc-algorithm': KeyInfo(default='aes-128,3des'),
                'hash-algorithm': KeyInfo(default='sha1'),
                'lifebytes': KeyInfo(can_disable=True, remove_value=0),
                'lifetime': KeyInfo(default='1d'),
                'name': KeyInfo(),
                'nat-traversal': KeyInfo(default=True),
                'prf-algorithm': KeyInfo(can_disable=True, remove_value='auto'),
                'proposal-check': KeyInfo(default='obey'),
            },
        ),
    ),
    ('ip', 'ipsec', 'proposal'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'auth-algorithms': KeyInfo(default='sha1'),
                'disabled': KeyInfo(default=False),
                'enc-algorithms': KeyInfo(default='aes-256-cbc,aes-192-cbc,aes-128-cbc'),
                'lifetime': KeyInfo(default='30m'),
                'name': KeyInfo(),
                'pfs-group': KeyInfo(default='modp1024'),
            },
        ),
    ),
    ('ip', 'pool'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'name': KeyInfo(),
                'ranges': KeyInfo(),
            },
        ),
    ),
    ('ip', 'route'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'blackhole': KeyInfo(can_disable=True),
                'check-gateway': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'distance': KeyInfo(),
                'dst-address': KeyInfo(),
                'gateway': KeyInfo(),
                'pref-src': KeyInfo(),
                'routing-table': KeyInfo(default='main'),
                'route-tag': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'scope': KeyInfo(),
                'suppress-hw-offload': KeyInfo(default=False),
                'target-scope': KeyInfo(),
                'type': KeyInfo(can_disable=True, remove_value='unicast'),
                'vrf-interface': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'route', 'vrf'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('routing-mark', ),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interfaces': KeyInfo(),
                'routing-mark': KeyInfo(),
            },
        ),
    ),
    ('ip', 'dhcp-server'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'address-pool': KeyInfo(default='static-only'),
                'allow-dual-stack-queue': KeyInfo(can_disable=True, remove_value=True),
                'always-broadcast': KeyInfo(can_disable=True, remove_value=False),
                'authoritative': KeyInfo(default=True),
                'bootp-lease-time': KeyInfo(default='forever'),
                'bootp-support': KeyInfo(can_disable=True, remove_value='static'),
                'client-mac-limit': KeyInfo(can_disable=True, remove_value='unlimited'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'conflict-detection': KeyInfo(can_disable=True, remove_value=True),
                'delay-threshold': KeyInfo(can_disable=True, remove_value='none'),
                'dhcp-option-set': KeyInfo(can_disable=True, remove_value='none'),
                'disabled': KeyInfo(default=False),
                'insert-queue-before': KeyInfo(can_disable=True, remove_value='first'),
                'interface': KeyInfo(required=True),
                'lease-script': KeyInfo(default=''),
                'lease-time': KeyInfo(default='10m'),
                'name': KeyInfo(),
                'parent-queue': KeyInfo(can_disable=True, remove_value='none'),
                'relay': KeyInfo(can_disable=True, remove_value='0.0.0.0'),
                'server-address': KeyInfo(can_disable=True, remove_value='0.0.0.0'),
                'use-framed-as-classless': KeyInfo(can_disable=True, remove_value=True),
                'use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('routing', 'filter', 'rule'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'chain': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(can_disable=True),
                    'rule': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('routing', 'filter', 'select-rule'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'chain': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(can_disable=True),
                    'do-group-num': KeyInfo(can_disable=True),
                    'do-group-prfx': KeyInfo(can_disable=True),
                    'do-jump': KeyInfo(can_disable=True),
                    'do-select-num': KeyInfo(can_disable=True),
                    'do-select-prfx': KeyInfo(can_disable=True),
                    'do-take': KeyInfo(can_disable=True),
                    'do-where': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('routing', 'ospf', 'instance'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'domain-id': KeyInfo(can_disable=True),
                'domain-tag': KeyInfo(can_disable=True),
                'in-filter-chain': KeyInfo(can_disable=True),
                'mpls-te-address': KeyInfo(can_disable=True),
                'mpls-te-area': KeyInfo(can_disable=True),
                'name': KeyInfo(),
                'originate-default': KeyInfo(can_disable=True),
                'out-filter-chain': KeyInfo(can_disable=True),
                'out-filter-select': KeyInfo(can_disable=True),
                'redistribute': KeyInfo(can_disable=True),
                'router-id': KeyInfo(default='main'),
                'routing-table': KeyInfo(can_disable=True),
                'use-dn': KeyInfo(can_disable=True),
                'version': KeyInfo(default=2),
                'vrf': KeyInfo(default='main'),
            },
        ),
    ),
    ('routing', 'ospf', 'area'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'area-id': KeyInfo(default='0.0.0.0'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-cost': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'instance': KeyInfo(required=True),
                'name': KeyInfo(),
                'no-summaries': KeyInfo(can_disable=True),
                'nssa-translator': KeyInfo(can_disable=True),
                'type': KeyInfo(default='default'),
            },
        ),
    ),
    ('routing', 'ospf', 'area', 'range'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('area', 'prefix', ),
            fields={
                'advertise': KeyInfo(default=True),
                'area': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'cost': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'prefix': KeyInfo(),
            },
        ),
    ),
    ('routing', 'ospf', 'interface-template'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'area': KeyInfo(required=True),
                'auth': KeyInfo(can_disable=True),
                'auth-id': KeyInfo(can_disable=True),
                'auth-key': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'cost': KeyInfo(default=1),
                'dead-interval': KeyInfo(default='40s'),
                'disabled': KeyInfo(default=False),
                'hello-interval': KeyInfo(default='10s'),
                'instance-id': KeyInfo(default=0),
                'interfaces': KeyInfo(can_disable=True),
                'networks': KeyInfo(can_disable=True),
                'passive': KeyInfo(can_disable=True),
                'prefix-list': KeyInfo(can_disable=True),
                'priority': KeyInfo(default=128),
                'retransmit-interval': KeyInfo(default='5s'),
                'transmit-delay': KeyInfo(default='1s'),
                'type': KeyInfo(default='broadcast'),
                'vlink-neighbor-id': KeyInfo(can_disable=True),
                'vlink-transit-area': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('routing', 'ospf-v3', 'instance'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'disabled': KeyInfo(),
                'distribute-default': KeyInfo(),
                'metric-bgp': KeyInfo(),
                'metric-connected': KeyInfo(),
                'metric-default': KeyInfo(),
                'metric-other-ospf': KeyInfo(),
                'metric-rip': KeyInfo(),
                'metric-static': KeyInfo(),
                'name': KeyInfo(),
                'redistribute-bgp': KeyInfo(),
                'redistribute-connected': KeyInfo(),
                'redistribute-other-ospf': KeyInfo(),
                'redistribute-rip': KeyInfo(),
                'redistribute-static': KeyInfo(),
                'router-id': KeyInfo(),
            },
        ),
    ),
    ('routing', 'ospf-v3', 'area'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'area-id': KeyInfo(),
                'disabled': KeyInfo(),
                'instance': KeyInfo(),
                'name': KeyInfo(),
                'type': KeyInfo(),
            },
        ),
    ),
    ('routing', 'pimsm', 'instance'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'afi': KeyInfo(default='ipv4'),
                'bsm-forward-back': KeyInfo(),
                'crp-advertise-contained': KeyInfo(),
                'disabled': KeyInfo(default=False),
                'name': KeyInfo(),
                'rp-hash-mask-length': KeyInfo(),
                'rp-static-override': KeyInfo(default=False),
                'ssm-range': KeyInfo(),
                'switch-to-spt': KeyInfo(default=True),
                'switch-to-spt-bytes': KeyInfo(default=0),
                'switch-to-spt-interval': KeyInfo(),
                'vrf': KeyInfo(default="main"),
            },
        ),
    ),
    ('routing', 'pimsm', 'interface-template'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=False),
                'hello-delay': KeyInfo(default='5s'),
                'hello-period': KeyInfo(default='30s'),
                'instance': KeyInfo(required=True),
                'interfaces': KeyInfo(can_disable=True),
                'join-prune-period': KeyInfo(default='1m'),
                'join-tracking-support': KeyInfo(default=True),
                'override-interval': KeyInfo(default='2s500ms'),
                'priority': KeyInfo(default=1),
                'propagation-delay': KeyInfo(default='500ms'),
                'source-addresses': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('routing', 'table'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'name': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(can_disable=True),
                    'fib': KeyInfo(),
                },
            )),
        ],
    ),
    ('snmp', 'community'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'addresses': KeyInfo(default='::/0'),
                'authentication-password': KeyInfo(default=''),
                'authentication-protocol': KeyInfo(default='MD5'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'encryption-password': KeyInfo(default=''),
                'encryption-protocol': KeyInfo(default='DES'),
                'name': KeyInfo(required=True),
                'read-access': KeyInfo(default=True),
                'security': KeyInfo(default='none'),
                'write-access': KeyInfo(default=False),
            },
        ),
    ),
    ('caps-man', 'aaa'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'called-format': KeyInfo(default='mac:ssid'),
                'interim-update': KeyInfo(default='disabled'),
                'mac-caching': KeyInfo(default='disabled'),
                'mac-format': KeyInfo(default='XX:XX:XX:XX:XX:XX'),
                'mac-mode': KeyInfo(default='as-username'),
            },
        ),
    ),
    ('caps-man', 'access-list'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'action': KeyInfo(can_disable=True),
                'allow-signal-out-of-range': KeyInfo(can_disable=True),
                'ap-tx-limit': KeyInfo(can_disable=True),
                'client-to-client-forwarding': KeyInfo(can_disable=True),
                'client-tx-limit': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(),
                'interface': KeyInfo(can_disable=True),
                'mac-address': KeyInfo(can_disable=True),
                'mac-address-mask': KeyInfo(can_disable=True),
                'private-passphrase': KeyInfo(can_disable=True),
                'radius-accounting': KeyInfo(can_disable=True),
                'signal-range': KeyInfo(can_disable=True),
                'ssid-regexp': KeyInfo(),
                'time': KeyInfo(can_disable=True),
                'vlan-id': KeyInfo(can_disable=True),
                'vlan-mode': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('caps-man', 'channel'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'band': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'control-channel-width': KeyInfo(can_disable=True),
                'extension-channel': KeyInfo(can_disable=True),
                'frequency': KeyInfo(can_disable=True),
                'name': KeyInfo(),
                'reselect-interval': KeyInfo(can_disable=True),
                'save-selected': KeyInfo(can_disable=True),
                'secondary-frequency': KeyInfo(can_disable=True),
                'skip-dfs-channels': KeyInfo(can_disable=True),
                'tx-power': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('caps-man', 'configuration'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'channel': KeyInfo(can_disable=True),
                'channel.band': KeyInfo(can_disable=True),
                'channel.control-channel-width': KeyInfo(can_disable=True),
                'channel.extension-channel': KeyInfo(can_disable=True),
                'channel.frequency': KeyInfo(can_disable=True),
                'channel.reselect-interval': KeyInfo(can_disable=True),
                'channel.save-selected': KeyInfo(can_disable=True),
                'channel.secondary-frequency': KeyInfo(can_disable=True),
                'channel.skip-dfs-channels': KeyInfo(can_disable=True),
                'channel.tx-power': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'country': KeyInfo(can_disable=True),
                'datapath': KeyInfo(can_disable=True),
                'datapath.arp': KeyInfo(),
                'datapath.bridge': KeyInfo(can_disable=True),
                'datapath.bridge-cost': KeyInfo(can_disable=True),
                'datapath.bridge-horizon': KeyInfo(can_disable=True),
                'datapath.client-to-client-forwarding': KeyInfo(can_disable=True),
                'datapath.interface-list': KeyInfo(can_disable=True),
                'datapath.l2mtu': KeyInfo(),
                'datapath.local-forwarding': KeyInfo(can_disable=True),
                'datapath.mtu': KeyInfo(),
                'datapath.openflow-switch': KeyInfo(can_disable=True),
                'datapath.vlan-id': KeyInfo(can_disable=True),
                'datapath.vlan-mode': KeyInfo(can_disable=True),
                'disconnect-timeout': KeyInfo(can_disable=True),
                'distance': KeyInfo(can_disable=True),
                'frame-lifetime': KeyInfo(can_disable=True),
                'guard-interval': KeyInfo(can_disable=True),
                'hide-ssid': KeyInfo(can_disable=True),
                'hw-protection-mode': KeyInfo(can_disable=True),
                'hw-retries': KeyInfo(can_disable=True),
                'installation': KeyInfo(can_disable=True),
                'keepalive-frames': KeyInfo(can_disable=True),
                'load-balancing-group': KeyInfo(can_disable=True),
                'max-sta-count': KeyInfo(can_disable=True),
                'mode': KeyInfo(can_disable=True),
                'multicast-helper': KeyInfo(can_disable=True),
                'name': KeyInfo(),
                'rates': KeyInfo(can_disable=True),
                'rates.basic': KeyInfo(can_disable=True),
                'rates.ht-basic-mcs': KeyInfo(can_disable=True),
                'rates.ht-supported-mcs': KeyInfo(can_disable=True),
                'rates.supported': KeyInfo(can_disable=True),
                'rates.vht-basic-mcs': KeyInfo(can_disable=True),
                'rates.vht-supported-mcs': KeyInfo(can_disable=True),
                'rx-chains': KeyInfo(can_disable=True),
                'security': KeyInfo(can_disable=True),
                'security.authentication-types': KeyInfo(can_disable=True),
                'security.disable-pmkid': KeyInfo(can_disable=True),
                'security.eap-methods': KeyInfo(can_disable=True),
                'security.eap-radius-accounting': KeyInfo(can_disable=True),
                'security.encryption': KeyInfo(can_disable=True),
                'security.group-encryption': KeyInfo(can_disable=True),
                'security.group-key-update': KeyInfo(),
                'security.passphrase': KeyInfo(can_disable=True),
                'security.tls-certificate': KeyInfo(),
                'security.tls-mode': KeyInfo(),
                'ssid': KeyInfo(can_disable=True),
                'tx-chains': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('caps-man', 'datapath'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'arp': KeyInfo(),
                'bridge': KeyInfo(can_disable=True),
                'bridge-cost': KeyInfo(can_disable=True),
                'bridge-horizon': KeyInfo(can_disable=True),
                'client-to-client-forwarding': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'interface-list': KeyInfo(can_disable=True),
                'l2mtu': KeyInfo(),
                'local-forwarding': KeyInfo(can_disable=True),
                'mtu': KeyInfo(),
                'name': KeyInfo(),
                'openflow-switch': KeyInfo(can_disable=True),
                'vlan-id': KeyInfo(can_disable=True),
                'vlan-mode': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('caps-man', 'manager', 'interface'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default': KeyInfo(),
                'disabled': KeyInfo(default=False),
                'forbid': KeyInfo(default=False),
                'interface': KeyInfo(),
            },
        ),
    ),
    ('caps-man', 'provisioning'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'action': KeyInfo(default='none'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'common-name-regexp': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'hw-supported-modes': KeyInfo(default=''),
                'identity-regexp': KeyInfo(default=''),
                'ip-address-ranges': KeyInfo(default=''),
                'master-configuration': KeyInfo(default='*FFFFFFFF'),
                'name-format': KeyInfo(default='cap'),
                'name-prefix': KeyInfo(default=''),
                'radio-mac': KeyInfo(default='00:00:00:00:00:00'),
                'slave-configurations': KeyInfo(default=''),
            },
        ),
    ),
    ('caps-man', 'security'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'authentication-types': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disable-pmkid': KeyInfo(can_disable=True),
                'eap-methods': KeyInfo(can_disable=True),
                'eap-radius-accounting': KeyInfo(can_disable=True),
                'encryption': KeyInfo(can_disable=True),
                'group-encryption': KeyInfo(can_disable=True),
                'group-key-update': KeyInfo(),
                'name': KeyInfo(),
                'passphrase': KeyInfo(can_disable=True),
                'tls-certificate': KeyInfo(),
                'tls-mode': KeyInfo(),
            }
        ),
    ),
    ('certificate', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'crl-download': KeyInfo(default=False),
                'crl-store': KeyInfo(default='ram'),
                'crl-use': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'bridge', 'port'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            fields={
                'auto-isolate': KeyInfo(default=False),
                'bpdu-guard': KeyInfo(default=False),
                'bridge': KeyInfo(required=True),
                'broadcast-flood': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'edge': KeyInfo(default='auto'),
                'fast-leave': KeyInfo(default=False),
                'frame-types': KeyInfo(default='admit-all'),
                'horizon': KeyInfo(default='none'),
                'hw': KeyInfo(default=True),
                'ingress-filtering': KeyInfo(default=True),
                'interface': KeyInfo(),
                'internal-path-cost': KeyInfo(default=10),
                'learn': KeyInfo(default='auto'),
                'multicast-router': KeyInfo(default='temporary-query'),
                'path-cost': KeyInfo(default=10),
                'point-to-point': KeyInfo(default='auto'),
                'priority': KeyInfo(default='0x80'),
                'pvid': KeyInfo(default=1),
                'restricted-role': KeyInfo(default=False),
                'restricted-tcn': KeyInfo(default=False),
                'tag-stacking': KeyInfo(default=False),
                'trusted': KeyInfo(default=False),
                'unknown-multicast-flood': KeyInfo(default=True),
                'unknown-unicast-flood': KeyInfo(default=True),
            },
        ),
    ),
    ('interface', 'bridge', 'mlag'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bridge': KeyInfo(default='none'),
                'peer-port': KeyInfo(default='none'),
            }
        ),
    ),
    ('interface', 'bridge', 'port-controller'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bridge': KeyInfo(default='none'),
                'cascade-ports': KeyInfo(default=''),
                'switch': KeyInfo(default='none'),
            },
        ),
    ),
    ('interface', 'bridge', 'port-extender'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'control-ports': KeyInfo(default=''),
                'excluded-ports': KeyInfo(default=''),
                'switch': KeyInfo(default='none'),
            },
        ),
    ),
    ('interface', 'bridge', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-fast-path': KeyInfo(default=True),
                'use-ip-firewall': KeyInfo(default=False),
                'use-ip-firewall-for-pppoe': KeyInfo(default=False),
                'use-ip-firewall-for-vlan': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'bridge', 'vlan'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('bridge', 'vlan-ids', ),
            fields={
                'bridge': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'tagged': KeyInfo(default=''),
                'untagged': KeyInfo(default=''),
                'vlan-ids': KeyInfo(),
            },
        ),
    ),
    ('ip', 'firewall', 'connection', 'tracking'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'enabled': KeyInfo(default='auto'),
                'generic-timeout': KeyInfo(default='10m'),
                'icmp-timeout': KeyInfo(default='10s'),
                'loose-tcp-tracking': KeyInfo(default=True),
                'tcp-close-timeout': KeyInfo(default='10s'),
                'tcp-close-wait-timeout': KeyInfo(default='10s'),
                'tcp-established-timeout': KeyInfo(default='1d'),
                'tcp-fin-wait-timeout': KeyInfo(default='10s'),
                'tcp-last-ack-timeout': KeyInfo(default='10s'),
                'tcp-max-retrans-timeout': KeyInfo(default='5m'),
                'tcp-syn-received-timeout': KeyInfo(default='5s'),
                'tcp-syn-sent-timeout': KeyInfo(default='5s'),
                'tcp-time-wait-timeout': KeyInfo(default='10s'),
                'tcp-unacked-timeout': KeyInfo(default='5m'),
                'udp-stream-timeout': KeyInfo(default='3m'),
                'udp-timeout': KeyInfo(default='10s'),
            },
        ),
    ),
    ('ip', 'neighbor', 'discovery-settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            versioned_fields=[
                ([('7.7', '>=')], 'mode', KeyInfo(default='tx-and-rx')),
            ],
            fields={
                'discover-interface-list': KeyInfo(),
                'lldp-med-net-policy-vlan': KeyInfo(default='disabled'),
                'protocol': KeyInfo(default='cdp,lldp,mndp'),
            },
        ),
    ),
    ('ip', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accept-redirects': KeyInfo(default=False),
                'accept-source-route': KeyInfo(default=False),
                'allow-fast-path': KeyInfo(default=True),
                'arp-timeout': KeyInfo(default='30s'),
                'icmp-rate-limit': KeyInfo(default=10),
                'icmp-rate-mask': KeyInfo(default='0x1818'),
                'ip-forward': KeyInfo(default=True),
                'max-neighbor-entries': KeyInfo(default=8192),
                'route-cache': KeyInfo(default=True),
                'rp-filter': KeyInfo(default=False),
                'secure-redirects': KeyInfo(default=True),
                'send-redirects': KeyInfo(default=True),
                'tcp-syncookies': KeyInfo(default=False),
            },
        ),
    ),
    ('ipv6', 'address'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'address': KeyInfo(),
                'advertise': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'eui-64': KeyInfo(default=False),
                'from-pool': KeyInfo(),
                'interface': KeyInfo(required=True),
                'no-dad': KeyInfo(default=False),
            },
        ),
    ),
    ('ipv6', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accept-redirects': KeyInfo(default='yes-if-forwarding-disabled'),
                'accept-router-advertisements': KeyInfo(default='yes-if-forwarding-disabled'),
                'disable-ipv6': KeyInfo(default=False),
                'forward': KeyInfo(default=True),
                'max-neighbor-entries': KeyInfo(default=8192),
            },
        ),
    ),
    ('interface', 'detect-internet'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'detect-interface-list': KeyInfo(default='none'),
                'internet-interface-list': KeyInfo(default='none'),
                'lan-interface-list': KeyInfo(default='none'),
                'wan-interface-list': KeyInfo(default='none'),
            },
        ),
    ),
    ('interface', 'l2tp-server', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-fast-path': KeyInfo(default=False),
                'authentication': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'caller-id-type': KeyInfo(default='ip-address'),
                'default-profile': KeyInfo(default='default-encryption'),
                'enabled': KeyInfo(default=False),
                'ipsec-secret': KeyInfo(default=''),
                'keepalive-timeout': KeyInfo(default=30),
                'max-mru': KeyInfo(default=1450),
                'max-mtu': KeyInfo(default=1450),
                'max-sessions': KeyInfo(default='unlimited'),
                'mrru': KeyInfo(default='disabled'),
                'one-session-per-host': KeyInfo(default=False),
                'use-ipsec': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'ovpn-server', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auth': KeyInfo(),
                'cipher': KeyInfo(),
                'default-profile': KeyInfo(default='default'),
                'enabled': KeyInfo(default=False),
                'keepalive-timeout': KeyInfo(default=60),
                'mac-address': KeyInfo(),
                'max-mtu': KeyInfo(default=1500),
                'mode': KeyInfo(default='ip'),
                'netmask': KeyInfo(default=24),
                'port': KeyInfo(default=1194),
                'require-client-certificate': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'pptp-server', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'authentication': KeyInfo(default='mschap1,mschap2'),
                'default-profile': KeyInfo(default='default-encryption'),
                'enabled': KeyInfo(default=False),
                'keepalive-timeout': KeyInfo(default=30),
                'max-mru': KeyInfo(default=1450),
                'max-mtu': KeyInfo(default=1450),
                'mrru': KeyInfo(default='disabled'),
            },
        ),
    ),
    ('interface', 'sstp-server', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'authentication': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'certificate': KeyInfo(default='none'),
                'default-profile': KeyInfo(default='default'),
                'enabled': KeyInfo(default=False),
                'force-aes': KeyInfo(default=False),
                'keepalive-timeout': KeyInfo(default=60),
                'max-mru': KeyInfo(default=1500),
                'max-mtu': KeyInfo(default=1500),
                'mrru': KeyInfo(default='disabled'),
                'pfs': KeyInfo(default=False),
                'port': KeyInfo(default=443),
                'tls-version': KeyInfo(default='any'),
                'verify-client-certificate': KeyInfo(default='no'),
            },
        ),
    ),
    ('interface', 'wireguard'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'listen-port': KeyInfo(),
                'mtu': KeyInfo(default=1420),
                'name': KeyInfo(),
                'private-key': KeyInfo(),
            },
        ),
    ),
    ('interface', 'wireguard', 'peers'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('public-key', 'interface'),
            fields={
                'allowed-address': KeyInfo(required=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'endpoint-address': KeyInfo(default=''),
                'endpoint-port': KeyInfo(default=0),
                'interface': KeyInfo(),
                'persistent-keepalive': KeyInfo(can_disable=True, remove_value=0),
                'preshared-key': KeyInfo(can_disable=True, remove_value=''),
                'public-key': KeyInfo(),
            },
        ),
    ),
    ('interface', 'wireless'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            required_one_of=[['default-name', 'master-interface']],
            fields={
                'adaptive-noise-immunity': KeyInfo(default='none'),
                'allow-sharedkey': KeyInfo(default=False),
                'ampdu-priorities': KeyInfo(default=0),
                'amsdu-limit': KeyInfo(default=8192),
                'amsdu-threshold': KeyInfo(default=8192),
                'antenna-gain': KeyInfo(default=0),
                'antenna-mode': KeyInfo(),
                'area': KeyInfo(default=''),
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'band': KeyInfo(),
                'basic-rates-a/g': KeyInfo(default='6Mbps'),
                'basic-rates-b': KeyInfo(default='1Mbps'),
                'bridge-mode': KeyInfo(default='enabled'),
                'channel-width': KeyInfo(default='20mhz'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'compression': KeyInfo(default=False),
                'country': KeyInfo(default='etsi'),
                'default-ap-tx-limit': KeyInfo(default=0),
                'default-authentication': KeyInfo(default=True),
                'default-client-tx-limit': KeyInfo(default=0),
                'default-forwarding': KeyInfo(default=True),
                'default-name': KeyInfo(),
                'disable-running-check': KeyInfo(default=False),
                'disabled': KeyInfo(default=True),
                'disconnect-timeout': KeyInfo(default='3s'),
                'distance': KeyInfo(default='dynamic'),
                'frame-lifetime': KeyInfo(default=0),
                'frequency': KeyInfo(),
                'frequency-mode': KeyInfo(default='regulatory-domain'),
                'frequency-offset': KeyInfo(default=0),
                'guard-interval': KeyInfo(default='any'),
                'hide-ssid': KeyInfo(default=False),
                'ht-basic-mcs': KeyInfo(),
                'ht-supported-mcs': KeyInfo(),
                'hw-fragmentation-threshold': KeyInfo(default='disabled'),
                'hw-protection-mode': KeyInfo(default='none'),
                'hw-protection-threshold': KeyInfo(default=0),
                'hw-retries': KeyInfo(default=7),
                'installation': KeyInfo(default='any'),
                'interworking-profile': KeyInfo(default='disabled'),
                'keepalive-frames': KeyInfo(default='enabled'),
                'l2mtu': KeyInfo(default=1600),
                'mac-address': KeyInfo(),
                'master-interface': KeyInfo(),
                'max-station-count': KeyInfo(default=2007),
                'mode': KeyInfo(default='ap-bridge'),
                'mtu': KeyInfo(default=1500),
                'multicast-buffering': KeyInfo(default='enabled'),
                'multicast-helper': KeyInfo(default='default'),
                'name': KeyInfo(),
                'noise-floor-threshold': KeyInfo(default='default'),
                'nv2-cell-radius': KeyInfo(default=30),
                'nv2-downlink-ratio': KeyInfo(default=50),
                'nv2-mode': KeyInfo(default='dynamic-downlink'),
                'nv2-noise-floor-offset': KeyInfo(default='default'),
                'nv2-preshared-key': KeyInfo(default=''),
                'nv2-qos': KeyInfo(default='default'),
                'nv2-queue-count': KeyInfo(default=2),
                'nv2-security': KeyInfo(default='disabled'),
                'nv2-sync-secret': KeyInfo(default=''),
                'on-fail-retry-time': KeyInfo(default='100ms'),
                'preamble-mode': KeyInfo(default='both'),
                'radio-name': KeyInfo(),
                'rate-selection': KeyInfo(default='advanced'),
                'rate-set': KeyInfo(default='default'),
                'running': KeyInfo(default=False),
                'rx-chains': KeyInfo(default='0,1'),
                'scan-list': KeyInfo(default='default'),
                'secondary-frequency': KeyInfo(default=''),
                'security-profile': KeyInfo(default='default'),
                'skip-dfs-channels': KeyInfo(default='disabled'),
                'ssid': KeyInfo(required=True),
                'station-bridge-clone-mac': KeyInfo(),
                'station-roaming': KeyInfo(default='disabled'),
                'supported-rates-a/g': KeyInfo(),
                'supported-rates-b': KeyInfo(),
                'tdma-period-size': KeyInfo(default=2),
                'tx-chains': KeyInfo(),
                'tx-power-mode': KeyInfo(default='default'),
                'update-stats-interval': KeyInfo(default='disabled'),
                'vlan-id': KeyInfo(default=1),
                'vlan-mode': KeyInfo(default='no-tag'),
                'wds-cost-range': KeyInfo(default='50-150'),
                'wds-default-bridge': KeyInfo(default='none'),
                'wds-default-cost': KeyInfo(default=100),
                'wds-ignore-ssid': KeyInfo(default=False),
                'wds-mode': KeyInfo(default='disabled'),
                'wireless-protocol': KeyInfo(default='any'),
                'wmm-support': KeyInfo(default='disabled'),
                'wps-mode': KeyInfo(default='push-button'),
            },
        ),
    ),
    ('interface', 'wireless', 'align'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'active-mode': KeyInfo(default=True),
                'audio-max': KeyInfo(default=-20),
                'audio-min': KeyInfo(default=-100),
                'audio-monitor': KeyInfo(default='00:00:00:00:00:00'),
                'filter-mac': KeyInfo(default='00:00:00:00:00:00'),
                'frame-size': KeyInfo(default=300),
                'frames-per-second': KeyInfo(default=25),
                'receive-all': KeyInfo(default=False),
                'ssid-all': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'wireless', 'cap'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bridge': KeyInfo(default='none'),
                'caps-man-addresses': KeyInfo(default=''),
                'caps-man-certificate-common-names': KeyInfo(default=''),
                'caps-man-names': KeyInfo(default=''),
                'certificate': KeyInfo(default='none'),
                'discovery-interfaces': KeyInfo(default=''),
                'enabled': KeyInfo(default=False),
                'interfaces': KeyInfo(default=''),
                'lock-to-caps-man': KeyInfo(default=False),
                'static-virtual': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'wireless', 'security-profiles'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'authentication-types': KeyInfo(),
                'disable-pmkid': KeyInfo(default=False),
                'disabled': KeyInfo(default=True),
                'eap-methods': KeyInfo(),
                'group-ciphers': KeyInfo(),
                'group-key-update': KeyInfo(default='5m'),
                'interim-update': KeyInfo(),
                'management-protection': KeyInfo(default='disabled'),
                'management-protection-key': KeyInfo(default=''),
                'mode': KeyInfo(default='none'),
                'mschapv2-password': KeyInfo(default=''),
                'mschapv2-username': KeyInfo(default=''),
                'name': KeyInfo(),
                'radius-called-format': KeyInfo(),
                'radius-eap-accounting': KeyInfo(default=False),
                'radius-mac-accounting': KeyInfo(default=False),
                'radius-mac-authentication': KeyInfo(default=False),
                'radius-mac-caching': KeyInfo(default='disabled'),
                'radius-mac-format': KeyInfo(default='XX:XX:XX:XX:XX:XX'),
                'radius-mac-mode': KeyInfo(default='as-username'),
                'static-algo-0': KeyInfo(default='none'),
                'static-algo-1': KeyInfo(default='none'),
                'static-algo-2': KeyInfo(default='none'),
                'static-algo-3': KeyInfo(default='none'),
                'static-key-0': KeyInfo(),
                'static-key-1': KeyInfo(),
                'static-key-2': KeyInfo(),
                'static-key-3': KeyInfo(),
                'static-sta-private-algo': KeyInfo(default='none'),
                'static-sta-private-key': KeyInfo(),
                'static-transmit-key': KeyInfo(),
                'supplicant-identity': KeyInfo(default='MikroTik'),
                'tls-certificate': KeyInfo(default='none'),
                'tls-mode': KeyInfo(),
                'unicast-ciphers': KeyInfo(),
                'wpa-pre-shared-key': KeyInfo(),
                'wpa2-pre-shared-key': KeyInfo(),
            },
        ),
    ),
    ('interface', 'wireless', 'sniffer'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'channel-time': KeyInfo(default='200ms'),
                'file-limit': KeyInfo(default=10),
                'file-name': KeyInfo(default=''),
                'memory-limit': KeyInfo(default=10),
                'multiple-channels': KeyInfo(default=False),
                'only-headers': KeyInfo(default=False),
                'receive-errors': KeyInfo(default=False),
                'streaming-enabled': KeyInfo(default=False),
                'streaming-max-rate': KeyInfo(default=0),
                'streaming-server': KeyInfo(default='0.0.0.0'),
            },
        ),
    ),
    ('interface', 'wireless', 'snooper'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'channel-time': KeyInfo(default='200ms'),
                'multiple-channels': KeyInfo(default=True),
                'receive-errors': KeyInfo(default=False),
            },
        ),
    ),
    ('iot', 'modbus'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=True),
                'hardware-port': KeyInfo(default='modbus'),
                'tcp-port': KeyInfo(default=502),
                'timeout': KeyInfo(default=1000),
            },
        ),
    ),
    ('ip', 'accounting'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'account-local-traffic': KeyInfo(default=False),
                'enabled': KeyInfo(default=False),
                'threshold': KeyInfo(default=256),
            },
        ),
    ),
    ('ip', 'accounting', 'web-access'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accessible-via-web': KeyInfo(default=False),
                'address': KeyInfo(default='0.0.0.0/0'),
            },
        ),
    ),
    ('ip', 'address'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('address', 'interface', ),
            fields={
                'address': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(),
                'network': KeyInfo(automatically_computed_from=('address', )),
            },
        ),
    ),
    ('ip', 'arp'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'address': KeyInfo(default='0.0.0.0'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(required=True),
                'mac-address': KeyInfo(default='00:00:00:00:00:00'),
                'published': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'cloud'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'ddns-enabled': KeyInfo(default=False),
                'ddns-update-interval': KeyInfo(default='none'),
                'update-time': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'cloud', 'advanced'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'use-local-address': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'dhcp-client'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            fields={
                'add-default-route': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-route-distance': KeyInfo(default=1),
                'dhcp-options': KeyInfo(default='hostname,clientid', can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(),
                'script': KeyInfo(can_disable=True),
                'use-peer-dns': KeyInfo(default=True),
                'use-peer-ntp': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'config'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accounting': KeyInfo(default=True),
                'interim-update': KeyInfo(default='0s'),
                'store-leases-disk': KeyInfo(default='5m'),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'lease'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('server', 'address', ),
            fields={
                'address': KeyInfo(),
                'address-lists': KeyInfo(default=''),
                'always-broadcast': KeyInfo(),
                'client-id': KeyInfo(can_disable=True, remove_value=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dhcp-option': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'insert-queue-before': KeyInfo(can_disable=True),
                'mac-address': KeyInfo(can_disable=True, remove_value=''),
                'server': KeyInfo(absent_value='all'),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'network'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('address', ),
            fields={
                'address': KeyInfo(),
                'boot-file-name': KeyInfo(default=''),
                'caps-manager': KeyInfo(default=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dhcp-option': KeyInfo(default=''),
                'dhcp-option-set': KeyInfo(default=''),
                'dns-none': KeyInfo(default=False),
                'dns-server': KeyInfo(default=''),
                'domain': KeyInfo(default=''),
                'gateway': KeyInfo(default=''),
                'netmask': KeyInfo(can_disable=True, remove_value=0),
                'next-server': KeyInfo(can_disable=True),
                'ntp-server': KeyInfo(default=''),
                'wins-server': KeyInfo(default=''),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'option'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'code': KeyInfo(required=True),
                'name': KeyInfo(),
                'value': KeyInfo(default=''),
                'force': KeyInfo(),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'option', 'sets'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'name': KeyInfo(required=True),
                'options': KeyInfo(),
            },
        ),
    ),
    ('ip', 'dns'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-remote-requests': KeyInfo(),
                'cache-max-ttl': KeyInfo(default='1w'),
                'cache-size': KeyInfo(default='2048KiB'),
                'max-concurrent-queries': KeyInfo(default=100),
                'max-concurrent-tcp-sessions': KeyInfo(default=20),
                'max-udp-packet-size': KeyInfo(default=4096),
                'query-server-timeout': KeyInfo(default='2s'),
                'query-total-timeout': KeyInfo(default='10s'),
                'servers': KeyInfo(default=''),
                'use-doh-server': KeyInfo(default=''),
                'verify-doh-cert': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'dns', 'static'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            required_one_of=[['name', 'regexp']],
            mutually_exclusive=[['name', 'regexp']],
            versioned_fields=[
                ([('7.5', '>=')], 'address-list', KeyInfo()),
                ([('7.5', '>=')], 'match-subdomain', KeyInfo(default=False)),
            ],
            fields={
                'address': KeyInfo(),
                'cname': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'forward-to': KeyInfo(),
                'mx-exchange': KeyInfo(),
                'mx-preference': KeyInfo(),
                'name': KeyInfo(),
                'ns': KeyInfo(),
                'regexp': KeyInfo(),
                'srv-port': KeyInfo(),
                'srv-priority': KeyInfo(),
                'srv-target': KeyInfo(),
                'srv-weight': KeyInfo(),
                'text': KeyInfo(),
                'ttl': KeyInfo(default='1d'),
                'type': KeyInfo(),
            },
        ),
    ),
    ('ip', 'firewall', 'address-list'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('address', 'list', ),
            fields={
                'address': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'list': KeyInfo(),
            },
        ),
    ),
    ('ip', 'firewall', 'filter'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-nat-state': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'fragment': KeyInfo(can_disable=True),
                'hotspot': KeyInfo(can_disable=True),
                'hw-offload': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'ipv4-options': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'layer7-protocol': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'p2p': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'psd': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'reject-with': KeyInfo(),
                'routing-mark': KeyInfo(can_disable=True),
                'routing-table': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'ttl': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'firewall', 'mangle'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-nat-state': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'fragment': KeyInfo(can_disable=True),
                'hotspot': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'ipv4-options': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'layer7-protocol': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'new-connection-mark': KeyInfo(can_disable=True),
                'new-dscp': KeyInfo(can_disable=True),
                'new-mss': KeyInfo(can_disable=True),
                'new-packet-mark': KeyInfo(can_disable=True),
                'new-priority': KeyInfo(can_disable=True),
                'new-routing-mark': KeyInfo(can_disable=True),
                'new-ttl': KeyInfo(can_disable=True),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'p2p': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'passthrough': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'psd': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'route-dst': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'routing-table': KeyInfo(can_disable=True),
                'sniff-id': KeyInfo(can_disable=True),
                'sniff-target': KeyInfo(can_disable=True),
                'sniff-target-port': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'ttl': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'firewall', 'nat'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(),
                'address-list-timeout': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'fragment': KeyInfo(can_disable=True),
                'hotspot': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'ipv4-options': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'layer7-protocol': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'psd': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'realm': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'same-not-by-dst': KeyInfo(),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'to-addresses': KeyInfo(can_disable=True),
                'to-ports': KeyInfo(can_disable=True),
                'ttl': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'firewall', 'raw'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain',),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(),
                'address-list-timeout': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'fragment': KeyInfo(can_disable=True),
                'hotspot': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'ipv4-options': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'psd': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'ttl': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'hotspot', 'user'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(),
                'name': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accounting': KeyInfo(default=True),
                'interim-update': KeyInfo(default='0s'),
                'xauth-use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'proxy'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'always-from-cache': KeyInfo(default=False),
                'anonymous': KeyInfo(default=False),
                'cache-administrator': KeyInfo(default='webmaster'),
                'cache-hit-dscp': KeyInfo(default=4),
                'cache-on-disk': KeyInfo(default=False),
                'cache-path': KeyInfo(default='web-proxy'),
                'enabled': KeyInfo(default=False),
                'max-cache-object-size': KeyInfo(default='2048KiB'),
                'max-cache-size': KeyInfo(default='unlimited'),
                'max-client-connections': KeyInfo(default=600),
                'max-fresh-time': KeyInfo(default='3d'),
                'max-server-connections': KeyInfo(default=600),
                'parent-proxy': KeyInfo(default='::'),
                'parent-proxy-port': KeyInfo(default=0),
                'port': KeyInfo(default=8080),
                'serialize-connections': KeyInfo(default=False),
                'src-address': KeyInfo(default='::'),
            },
        ),
    ),
    ('ip', 'smb'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-guests': KeyInfo(default=True),
                'comment': KeyInfo(default='MikrotikSMB'),
                'domain': KeyInfo(default='MSHOME'),
                'enabled': KeyInfo(default=False),
                'interfaces': KeyInfo(default='all'),
            },
        ),
    ),
    ('ip', 'smb', 'shares'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'directory': KeyInfo(),
                'disabled': KeyInfo(),
                'max-sessions': KeyInfo(),
                'name': KeyInfo(),
            },
        ),
    ),
    ('ip', 'smb', 'users'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'disabled': KeyInfo(),
                'name': KeyInfo(),
                'password': KeyInfo(),
                'read-only': KeyInfo(),
            },
        ),
    ),
    ('ip', 'socks'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auth-method': KeyInfo(default='none'),
                'connection-idle-timeout': KeyInfo(default='2m'),
                'enabled': KeyInfo(default=False),
                'max-connections': KeyInfo(default=200),
                'port': KeyInfo(default=1080),
                'version': KeyInfo(default=4),
            },
        ),
    ),
    ('ip', 'ssh'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-none-crypto': KeyInfo(default=False),
                'always-allow-password-login': KeyInfo(default=False),
                'forwarding-enabled': KeyInfo(default=False),
                'host-key-size': KeyInfo(default=2048),
                'strong-crypto': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'tftp', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'max-block-size': KeyInfo(default=4096),
            },
        ),
    ),
    ('ip', 'traffic-flow'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'active-flow-timeout': KeyInfo(default='30m'),
                'cache-entries': KeyInfo(default='32k'),
                'enabled': KeyInfo(default=False),
                'inactive-flow-timeout': KeyInfo(default='15s'),
                'interfaces': KeyInfo(default='all'),
                'packet-sampling': KeyInfo(default=False),
                'sampling-interval': KeyInfo(default=0),
                'sampling-space': KeyInfo(default=0),
            },
        ),
    ),
    ('ip', 'traffic-flow', 'ipfix'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bytes': KeyInfo(default=True),
                'dst-address': KeyInfo(default=True),
                'dst-address-mask': KeyInfo(default=True),
                'dst-mac-address': KeyInfo(default=True),
                'dst-port': KeyInfo(default=True),
                'first-forwarded': KeyInfo(default=True),
                'gateway': KeyInfo(default=True),
                'icmp-code': KeyInfo(default=True),
                'icmp-type': KeyInfo(default=True),
                'igmp-type': KeyInfo(default=True),
                'in-interface': KeyInfo(default=True),
                'ip-header-length': KeyInfo(default=True),
                'ip-total-length': KeyInfo(default=True),
                'ipv6-flow-label': KeyInfo(default=True),
                'is-multicast': KeyInfo(default=True),
                'last-forwarded': KeyInfo(default=True),
                'nat-dst-address': KeyInfo(default=True),
                'nat-dst-port': KeyInfo(default=True),
                'nat-events': KeyInfo(default=False),
                'nat-src-address': KeyInfo(default=True),
                'nat-src-port': KeyInfo(default=True),
                'out-interface': KeyInfo(default=True),
                'packets': KeyInfo(default=True),
                'protocol': KeyInfo(default=True),
                'src-address': KeyInfo(default=True),
                'src-address-mask': KeyInfo(default=True),
                'src-mac-address': KeyInfo(default=True),
                'src-port': KeyInfo(default=True),
                'sys-init-time': KeyInfo(default=True),
                'tcp-ack-num': KeyInfo(default=True),
                'tcp-flags': KeyInfo(default=True),
                'tcp-seq-num': KeyInfo(default=True),
                'tcp-window-size': KeyInfo(default=True),
                'tos': KeyInfo(default=True),
                'ttl': KeyInfo(default=True),
                'udp-length': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'traffic-flow', 'target'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'address': KeyInfo(),
                'disabled': KeyInfo(default=False),
                'dst-address': KeyInfo(),
                'port': KeyInfo(default=2055),
                'src-address': KeyInfo(),
                'v9-template-refresh': KeyInfo(default=20),
                'v9-template-timeout': KeyInfo(),
                'version': KeyInfo(),
            },
        ),
    ),
    ('ip', 'upnp'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-disable-external-interface': KeyInfo(default=False),
                'enabled': KeyInfo(default=False),
                'show-dummy-rule': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'upnp', 'interfaces'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', 'type'),
            fields={
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(),
                'type': KeyInfo(),
                'forced-ip': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ipv6', 'dhcp-client'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', 'request'),
            fields={
                'add-default-route': KeyInfo(default=False),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-route-distance': KeyInfo(default=1),
                'dhcp-options': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(),
                'pool-name': KeyInfo(required=True),
                'pool-prefix-length': KeyInfo(default=64),
                'prefix-hint': KeyInfo(default='::/0'),
                'request': KeyInfo(),
                'use-peer-dns': KeyInfo(default=True),
            },
        ),
    ),
    ('ipv6', 'dhcp-server'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'address-pool': KeyInfo(required=True),
                'allow-dual-stack-queue': KeyInfo(can_disable=True, remove_value=True),
                'binding-script': KeyInfo(can_disable=True, remove_value=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dhcp-option': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'insert-queue-before': KeyInfo(can_disable=True, remove_value='first'),
                'interface': KeyInfo(required=True),
                'lease-time': KeyInfo(default='3d'),
                'name': KeyInfo(),
                'parent-queue': KeyInfo(can_disable=True, remove_value='none'),
                'preference': KeyInfo(default=255),
                'rapid-commit': KeyInfo(default=True),
                'route-distance': KeyInfo(default=1),
                'use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('ipv6', 'dhcp-server', 'option'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'code': KeyInfo(required=True),
                'name': KeyInfo(),
                'value': KeyInfo(default=''),
            },
        ),
    ),
    ('ipv6', 'firewall', 'address-list'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('address', 'list', ),
            fields={
                'address': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'list': KeyInfo(),
            },
        ),
    ),
    ('ipv6', 'firewall', 'filter'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'headers': KeyInfo(can_disable=True),
                'hop-limit': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'reject-with': KeyInfo(),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ipv6', 'firewall', 'mangle'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(),
                'address-list-timeout': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'dst-prefix': KeyInfo(),
                'headers': KeyInfo(can_disable=True),
                'hop-limit': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'new-connection-mark': KeyInfo(),
                'new-dscp': KeyInfo(),
                'new-hop-limit': KeyInfo(),
                'new-mss': KeyInfo(),
                'new-packet-mark': KeyInfo(),
                'new-routing-mark': KeyInfo(),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'passthrough': KeyInfo(),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'sniff-id': KeyInfo(),
                'sniff-target': KeyInfo(),
                'sniff-target-port': KeyInfo(),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'src-prefix': KeyInfo(),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
            }
        ),
    ),
    ('ipv6', 'firewall', 'nat'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(),
                'address-list-timeout': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'layer7-protocol': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'to-addresses': KeyInfo(can_disable=True),
                'to-ports': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ipv6', 'firewall', 'raw'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain',),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(),
                'address-list-timeout': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'headers': KeyInfo(can_disable=True),
                'hop-limit': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
            }
        ),
    ),
    ('ipv6', 'nd'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            fields={
                'advertise-dns': KeyInfo(default=True),
                'advertise-mac-address': KeyInfo(default=True),
                'disabled': KeyInfo(default=False),
                'dns': KeyInfo(default=''),
                'hop-limit': KeyInfo(default='unspecified'),
                'interface': KeyInfo(),
                'managed-address-configuration': KeyInfo(default=False),
                'mtu': KeyInfo(default='unspecified'),
                'other-configuration': KeyInfo(default=False),
                'ra-delay': KeyInfo(default='3s'),
                'ra-interval': KeyInfo(default='3m20s-10m'),
                'ra-lifetime': KeyInfo(default='30m'),
                'ra-preference': KeyInfo(default='medium'),
                'reachable-time': KeyInfo(default='unspecified'),
                'retransmit-interval': KeyInfo(default='unspecified'),
            },
        ),
    ),
    ('ipv6', 'nd', 'prefix', 'default'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'autonomous': KeyInfo(default=True),
                'preferred-lifetime': KeyInfo(default='1w'),
                'valid-lifetime': KeyInfo(default='4w2d'),
            },
        ),
    ),
    ('ipv6', 'route'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'bgp-as-path': KeyInfo(can_disable=True),
                'bgp-atomic-aggregate': KeyInfo(can_disable=True),
                'bgp-communities': KeyInfo(can_disable=True),
                'bgp-local-pref': KeyInfo(can_disable=True),
                'bgp-med': KeyInfo(can_disable=True),
                'bgp-origin': KeyInfo(can_disable=True),
                'bgp-prepend': KeyInfo(can_disable=True),
                'type': KeyInfo(can_disable=True, remove_value='unicast'),
                'blackhole': KeyInfo(can_disable=True),
                'check-gateway': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(),
                'distance': KeyInfo(default=1),
                'dst-address': KeyInfo(),
                'gateway': KeyInfo(),
                'route-tag': KeyInfo(can_disable=True),
                'routing-table': KeyInfo(default='main'),
                'scope': KeyInfo(default=30),
                'target-scope': KeyInfo(default=10),
                'vrf-interface': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('mpls', ): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-fast-path': KeyInfo(default=True),
                'dynamic-label-range': KeyInfo(default='16-1048575'),
                'propagate-ttl': KeyInfo(default=True),
            },
        ),
    ),
    ('mpls', 'interface'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'disabled': KeyInfo(),
                'interface': KeyInfo(),
                'mpls-mtu': KeyInfo(),
            },
        ),
    ),
    ('mpls', 'ldp'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'distribute-for-default-route': KeyInfo(default=False),
                'enabled': KeyInfo(default=False),
                'hop-limit': KeyInfo(default=255),
                'loop-detect': KeyInfo(default=False),
                'lsr-id': KeyInfo(default='0.0.0.0'),
                'path-vector-limit': KeyInfo(default=255),
                'transport-address': KeyInfo(default='0.0.0.0'),
                'use-explicit-null': KeyInfo(default=False),
            },
        ),
    ),
    ('port', 'firmware'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'directory': KeyInfo(default='firmware'),
                'ignore-directip-modem': KeyInfo(default=False),
            },
        ),
    ),
    ('port', 'remote-access'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'allowed-addresses': KeyInfo(default='0.0.0.0/0'),
                'channel': KeyInfo(default=0),
                'disabled': KeyInfo(default=False),
                'log-file': KeyInfo(default=""),
                'port': KeyInfo(required=True),
                'protocol': KeyInfo(default='rfc2217'),
                'tcp-port': KeyInfo(default=0),
            },
        ),
    ),
    ('ppp', 'aaa'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accounting': KeyInfo(default=True),
                'interim-update': KeyInfo(default='0s'),
                'use-circuit-id-in-nas-port-id': KeyInfo(default=False),
                'use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('radius', 'incoming'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accept': KeyInfo(default=False),
                'port': KeyInfo(default=3799),
            },
        ),
    ),
    ('routing', 'id'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=False),
                'id': KeyInfo(),
                'name': KeyInfo(),
                'select-dynamic-id': KeyInfo(),
                'select-from-vrf': KeyInfo(),
            },
        ),
    ),
    ('routing', 'bfd', 'interface'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'disabled': KeyInfo(),
                'interface': KeyInfo(),
                'interval': KeyInfo(),
                'min-rx': KeyInfo(),
                'multiplier': KeyInfo(),
            },
        ),
    ),
    ('routing', 'mme'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bidirectional-timeout': KeyInfo(default=2),
                'gateway-class': KeyInfo(default='none'),
                'gateway-keepalive': KeyInfo(default='1m'),
                'gateway-selection': KeyInfo(default='no-gateway'),
                'origination-interval': KeyInfo(default='5s'),
                'preferred-gateway': KeyInfo(default='0.0.0.0'),
                'timeout': KeyInfo(default='1m'),
                'ttl': KeyInfo(default=50),
            },
        ),
    ),
    ('routing', 'rip'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'distribute-default': KeyInfo(default='never'),
                'garbage-timer': KeyInfo(default='2m'),
                'metric-bgp': KeyInfo(default=1),
                'metric-connected': KeyInfo(default=1),
                'metric-default': KeyInfo(default=1),
                'metric-ospf': KeyInfo(default=1),
                'metric-static': KeyInfo(default=1),
                'redistribute-bgp': KeyInfo(default=False),
                'redistribute-connected': KeyInfo(default=False),
                'redistribute-ospf': KeyInfo(default=False),
                'redistribute-static': KeyInfo(default=False),
                'routing-table': KeyInfo(default='main'),
                'timeout-timer': KeyInfo(default='3m'),
                'update-timer': KeyInfo(default='30s'),
            },
        ),
    ),
    ('routing', 'ripng'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'distribute-default': KeyInfo(default='never'),
                'garbage-timer': KeyInfo(default='2m'),
                'metric-bgp': KeyInfo(default=1),
                'metric-connected': KeyInfo(default=1),
                'metric-default': KeyInfo(default=1),
                'metric-ospf': KeyInfo(default=1),
                'metric-static': KeyInfo(default=1),
                'redistribute-bgp': KeyInfo(default=False),
                'redistribute-connected': KeyInfo(default=False),
                'redistribute-ospf': KeyInfo(default=False),
                'redistribute-static': KeyInfo(default=False),
                'timeout-timer': KeyInfo(default='3m'),
                'update-timer': KeyInfo(default='30s'),
            },
        ),
    ),
    ('snmp', ): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'contact': KeyInfo(default=''),
                'enabled': KeyInfo(default=False),
                'location': KeyInfo(default=''),
                'src-address': KeyInfo(default='::'),
                'trap-community': KeyInfo(default='public'),
                'trap-generators': KeyInfo(default='temp-exception'),
                'trap-target': KeyInfo(default=''),
                'trap-version': KeyInfo(default=1),
                'trap-interfaces': KeyInfo(default=''),
            },
            versioned_fields=[
                ([('7.10', '<')], 'engine-id', KeyInfo(default='')),
                ([('7.10', '>=')], 'engine-id', KeyInfo(read_only=True)),
                ([('7.10', '>=')], 'engine-id-suffix', KeyInfo(default='')),
            ],
        ),
    ),
    ('system', 'clock'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'date': KeyInfo(),
                'gmt-offset': KeyInfo(),
                'time': KeyInfo(),
                'time-zone-autodetect': KeyInfo(default=True),
                'time-zone-name': KeyInfo(default='manual'),
            },
        ),
    ),
    ('system', 'clock', 'manual'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'dst-delta': KeyInfo(default='00:00'),
                'dst-end': KeyInfo(default='jan/01/1970 00:00:00'),
                'dst-start': KeyInfo(default='jan/01/1970 00:00:00'),
                'time-zone': KeyInfo(default='+00:00'),
            },
        ),
    ),
    ('system', 'identity'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'name': KeyInfo(default='Mikrotik'),
            },
        ),
    ),
    ('system', 'leds', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'all-leds-off': KeyInfo(default='never'),
            },
        ),
    ),
    ('system', 'note'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'note': KeyInfo(default=''),
                'show-at-login': KeyInfo(default=True),
            },
        ),
    ),
    ('system', 'ntp', 'client'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'enabled': KeyInfo(default=False),
                'primary-ntp': KeyInfo(default='0.0.0.0'),
                'secondary-ntp': KeyInfo(default='0.0.0.0'),
                'server-dns-names': KeyInfo(default=''),
                'servers': KeyInfo(default=''),
                'mode': KeyInfo(default='unicast'),
                'vrf': KeyInfo(default='main'),
            },
        ),
    ),
    ('system', 'ntp', 'client', 'servers'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('address', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'address': KeyInfo(),
                'auth-key': KeyInfo(default='none'),
                'iburst': KeyInfo(default=True),
                'max-poll': KeyInfo(default=10),
                'min-poll': KeyInfo(default=6),
            },
        ),
    ),
    ('system', 'ntp', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auth-key': KeyInfo(default='none'),
                'broadcast': KeyInfo(default=False),
                'broadcast-addresses': KeyInfo(default=''),
                'enabled': KeyInfo(default=False),
                'local-clock-stratum': KeyInfo(default=5),
                'manycast': KeyInfo(default=False),
                'multicast': KeyInfo(default=False),
                'use-local-clock': KeyInfo(default=False),
                'vrf': KeyInfo(default='main'),
            },
        ),
    ),
    ('system', 'package', 'update'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'channel': KeyInfo(default='stable'),
            },
        ),
    ),
    ('system', 'routerboard', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auto-upgrade': KeyInfo(default=False),
                'baud-rate': KeyInfo(default=115200),
                'boot-delay': KeyInfo(default='2s'),
                'boot-device': KeyInfo(default='nand-if-fail-then-ethernet'),
                'boot-protocol': KeyInfo(default='bootp'),
                'enable-jumper-reset': KeyInfo(default=True),
                'enter-setup-on': KeyInfo(default='any-key'),
                'force-backup-booter': KeyInfo(default=False),
                'protected-routerboot': KeyInfo(default='disabled'),
                'reformat-hold-button': KeyInfo(default='20s'),
                'reformat-hold-button-max': KeyInfo(default='10m'),
                'silent-boot': KeyInfo(default=False),
            },
        ),
    ),
    ('system', 'upgrade', 'mirror'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'check-interval': KeyInfo(default='1d'),
                'enabled': KeyInfo(default=False),
                'primary-server': KeyInfo(default='0.0.0.0'),
                'secondary-server': KeyInfo(default='0.0.0.0'),
                'user': KeyInfo(default=''),
            },
        ),
    ),
    ('system', 'ups'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'alarm-setting': KeyInfo(default='immediate'),
                'check-capabilities': KeyInfo(can_disable=True, remove_value=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=True),
                'min-runtime': KeyInfo(default='never'),
                'name': KeyInfo(),
                'offline-time': KeyInfo(default='0s'),
                'port': KeyInfo(required=True),
            },
        ),
    ),
    ('system', 'watchdog'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auto-send-supout': KeyInfo(default=False),
                'automatic-supout': KeyInfo(default=True),
                'ping-start-after-boot': KeyInfo(default='5m'),
                'ping-timeout': KeyInfo(default='1m'),
                'watch-address': KeyInfo(default='none'),
                'watchdog-timer': KeyInfo(default=True),
            },
        ),
    ),
    ('tool', 'bandwidth-server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allocate-udp-ports-from': KeyInfo(default=2000),
                'authenticate': KeyInfo(default=True),
                'enabled': KeyInfo(default=True),
                'max-sessions': KeyInfo(default=100),
            },
        ),
    ),
    ('tool', 'e-mail'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'address': KeyInfo(default='0.0.0.0'),
                'from': KeyInfo(default='<>'),
                'password': KeyInfo(default=''),
                'port': KeyInfo(default=25),
                'start-tls': KeyInfo(default=False),
                'tls': KeyInfo(default=False),
                'user': KeyInfo(default=''),
            },
        ),
    ),
    ('tool', 'graphing'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'page-refresh': KeyInfo(default=300),
                'store-every': KeyInfo(default='5min'),
            },
        ),
    ),
    ('tool', 'graphing', 'interface'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'allow-address': KeyInfo(default='0.0.0.0/0'),
                    'interface': KeyInfo(default='all'),
                    'store-on-disk': KeyInfo(default=True),
                },
            )),
        ],
    ),
    ('tool', 'graphing', 'resource'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'allow-address': KeyInfo(default='0.0.0.0/0'),
                    'store-on-disk': KeyInfo(default=True),
                },
            )),
        ],
    ),
    ('tool', 'mac-server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allowed-interface-list': KeyInfo(),
            },
        ),
    ),
    ('tool', 'mac-server', 'mac-winbox'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allowed-interface-list': KeyInfo(),
            },
        ),
    ),
    ('tool', 'mac-server', 'ping'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'enabled': KeyInfo(default=True),
            },
        ),
    ),
    ('tool', 'netwatch'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'certificate': KeyInfo(),
                    'check-certificate': KeyInfo(),
                    'comment': KeyInfo(),
                    'disabled': KeyInfo(default=False),
                    'down-script': KeyInfo(),
                    'host': KeyInfo(required=True),
                    'http-codes': KeyInfo(),
                    'interval': KeyInfo(),
                    'name': KeyInfo(),
                    'packet-count': KeyInfo(),
                    'packet-interval': KeyInfo(),
                    'packet-size': KeyInfo(),
                    'port': KeyInfo(),
                    'src-address': KeyInfo(),
                    'start-delay': KeyInfo(),
                    'startup-delay': KeyInfo(),
                    'test-script': KeyInfo(),
                    'thr-avg': KeyInfo(),
                    'thr-http-time': KeyInfo(),
                    'thr-jitter': KeyInfo(),
                    'thr-loss-count': KeyInfo(),
                    'thr-loss-percent': KeyInfo(),
                    'thr-max': KeyInfo(),
                    'thr-stdev': KeyInfo(),
                    'thr-tcp-conn-time': KeyInfo(),
                    'timeout': KeyInfo(),
                    'type': KeyInfo(default='simple'),
                    'up-script': KeyInfo(),
                },
            )),
        ],
    ),
    ('tool', 'romon'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'enabled': KeyInfo(default=False),
                'id': KeyInfo(default='00:00:00:00:00:00'),
                'secrets': KeyInfo(default=''),
            },
        ),
    ),
    ('tool', 'romon', 'port'): APIData(
        unversioned=VersionedAPIData(
            fields={
                'cost': KeyInfo(),
                'disabled': KeyInfo(),
                'forbid': KeyInfo(),
                'interface': KeyInfo(),
                'secrets': KeyInfo(),
            },
        ),
    ),
    ('tool', 'sms'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allowed-number': KeyInfo(default=''),
                'auto-erase': KeyInfo(default=False),
                'channel': KeyInfo(default=0),
                'port': KeyInfo(default='none'),
                'receive-enabled': KeyInfo(default=False),
                'secret': KeyInfo(default=''),
                'sim-pin': KeyInfo(default=''),
            },
        ),
    ),
    ('tool', 'sniffer'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'file-limit': KeyInfo(default='1000KiB'),
                'file-name': KeyInfo(default=''),
                'filter-cpu': KeyInfo(default=''),
                'filter-direction': KeyInfo(default='any'),
                'filter-interface': KeyInfo(default=''),
                'filter-ip-address': KeyInfo(default=''),
                'filter-ip-protocol': KeyInfo(default=''),
                'filter-ipv6-address': KeyInfo(default=''),
                'filter-mac-address': KeyInfo(default=''),
                'filter-mac-protocol': KeyInfo(default=''),
                'filter-operator-between-entries': KeyInfo(default='or'),
                'filter-port': KeyInfo(default=''),
                'filter-size': KeyInfo(default=''),
                'filter-stream': KeyInfo(default=False),
                'memory-limit': KeyInfo(default='100KiB'),
                'memory-scroll': KeyInfo(default=True),
                'only-headers': KeyInfo(default=False),
                'streaming-enabled': KeyInfo(default=False),
                'streaming-server': KeyInfo(default='0.0.0.0:37008'),
            },
        ),
    ),
    ('tool', 'traffic-generator'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'latency-distribution-max': KeyInfo(default='100us'),
                'measure-out-of-order': KeyInfo(default=True),
                'stats-samples-to-keep': KeyInfo(default=100),
                'test-id': KeyInfo(default=0),
            },
        ),
    ),
    ('user',): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'address': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'expired': KeyInfo(read_only=True),
                'group': KeyInfo(),
                'last-logged-in': KeyInfo(read_only=True),
                'name': KeyInfo(),
                'password': KeyInfo(write_only=True),
            },
        ),
    ),
    ('user', 'aaa'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accounting': KeyInfo(default=True),
                'default-group': KeyInfo(default='read'),
                'exclude-groups': KeyInfo(default=''),
                'interim-update': KeyInfo(default='0s'),
                'use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('user', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'minimum-categories': KeyInfo(),
                'minimum-password-length': KeyInfo(),
            },
        ),
    ),
    ('queue', 'interface'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('interface', ),
            fully_understood=True,
            fixed_entries=True,
            fields={
                'interface': KeyInfo(required=True),
                'queue': KeyInfo(required=True),
            },
        ),
    ),
    ('queue', 'tree'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'bucket-size': KeyInfo(default='0.1'),
                'burst-limit': KeyInfo(default=0),
                'burst-threshold': KeyInfo(default=0),
                'burst-time': KeyInfo(default='0s'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'limit-at': KeyInfo(default=0),
                'max-limit': KeyInfo(default=0),
                'name': KeyInfo(),
                'packet-mark': KeyInfo(default=''),
                'parent': KeyInfo(required=True),
                'priority': KeyInfo(default=8),
                'queue': KeyInfo(default='default-small'),
            },
        ),
    ),
    ('interface', 'ethernet', 'switch'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'cpu-flow-control': KeyInfo(default=True),
                'mirror-source': KeyInfo(default='none'),
                'mirror-target': KeyInfo(default='none'),
                'name': KeyInfo(),
            },
        ),
    ),
    ('interface', 'ethernet', 'switch', 'port'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'default-vlan-id': KeyInfo(),
                'name': KeyInfo(),
                'vlan-header': KeyInfo(default='leave-as-is'),
                'vlan-mode': KeyInfo(default='disabled'),
            },
        ),
    ),
    ('ip', 'dhcp-client', 'option'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'code': KeyInfo(),
                'name': KeyInfo(),
                'value': KeyInfo(),
            },
        ),
    ),
    ('ppp', 'profile'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'address-list': KeyInfo(default=''),
                'bridge': KeyInfo(can_disable=True),
                'bridge-horizon': KeyInfo(can_disable=True),
                'bridge-learning': KeyInfo(default='default'),
                'bridge-path-cost': KeyInfo(can_disable=True),
                'bridge-port-priority': KeyInfo(can_disable=True),
                'change-tcp-mss': KeyInfo(default=True),
                'dns-server': KeyInfo(can_disable=True),
                'idle-timeout': KeyInfo(can_disable=True),
                'incoming-filter': KeyInfo(can_disable=True),
                'insert-queue-before': KeyInfo(can_disable=True),
                'interface-list': KeyInfo(can_disable=True),
                'local-address': KeyInfo(can_disable=True),
                'name': KeyInfo(required=True),
                'on-down': KeyInfo(default=''),
                'on-up': KeyInfo(default=''),
                'only-one': KeyInfo(default='default'),
                'outgoing-filter': KeyInfo(can_disable=True),
                'parent-queue': KeyInfo(can_disable=True),
                'queue-type': KeyInfo(can_disable=True),
                'rate-limit': KeyInfo(can_disable=True),
                'remote-address': KeyInfo(can_disable=True),
                'session-timeout': KeyInfo(can_disable=True),
                'use-compression': KeyInfo(default='default'),
                'use-encryption': KeyInfo(default='default'),
                'use-ipv6': KeyInfo(default=True),
                'use-mpls': KeyInfo(default='default'),
                'use-upnp': KeyInfo(default='default'),
                'wins-server': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('queue', 'type'): APIData(
        unversioned=VersionedAPIData(
            has_identifier=True,
            fields={
                'kind': KeyInfo(),
                'mq-pfifo-limit': KeyInfo(),
                'name': KeyInfo(),
                'pcq-burst-rate': KeyInfo(),
                'pcq-burst-threshold': KeyInfo(),
                'pcq-burst-time': KeyInfo(),
                'pcq-classifier': KeyInfo(),
                'pcq-dst-address-mask': KeyInfo(),
                'pcq-dst-address6-mask': KeyInfo(),
                'pcq-limit': KeyInfo(),
                'pcq-rate': KeyInfo(),
                'pcq-src-address-mask': KeyInfo(),
                'pcq-src-address6-mask': KeyInfo(),
                'pcq-total-limit': KeyInfo(),
                'pfifo-limit': KeyInfo(),
                'red-avg-packet': KeyInfo(),
                'red-burst': KeyInfo(),
                'red-limit': KeyInfo(),
                'red-max-threshold': KeyInfo(),
                'red-min-threshold': KeyInfo(),
                'sfq-allot': KeyInfo(),
                'sfq-perturb': KeyInfo(),
            },
        ),
    ),
    ('routing', 'bgp', 'connection'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'as': KeyInfo(),
                'name': KeyInfo(required=True),
                'connect': KeyInfo(default=True),
                'listen': KeyInfo(default=True),
                'local.address': KeyInfo(),
                'local.port': KeyInfo(),
                'local.role': KeyInfo(required=True),
                'local.ttl': KeyInfo(),
                'remote.address': KeyInfo(required=True),
                'remote.port': KeyInfo(),
                'remote.as': KeyInfo(),
                'remote.allowed-as': KeyInfo(),
                'remote.ttl': KeyInfo(),
                'tcp-md5-key': KeyInfo(),
                'templates': KeyInfo(),
            },
        ),
    ),
    ('routing', 'bgp', 'instance'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'as': KeyInfo(),
                'client-to-client-reflection': KeyInfo(),
                'cluster-id': KeyInfo(can_disable=True),
                'confederation': KeyInfo(can_disable=True),
                'disabled': KeyInfo(),
                'ignore-as-path-len': KeyInfo(),
                'name': KeyInfo(),
                'out-filter': KeyInfo(),
                'redistribute-connected': KeyInfo(),
                'redistribute-ospf': KeyInfo(),
                'redistribute-other-bgp': KeyInfo(),
                'redistribute-rip': KeyInfo(),
                'redistribute-static': KeyInfo(),
                'router-id': KeyInfo(),
                'routing-table': KeyInfo(),
            },
        ),
    ),
    ('system', 'logging', 'action'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'bsd-syslog': KeyInfo(default=False),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disk-file-count': KeyInfo(default=2),
                'disk-file-name': KeyInfo(default='log'),
                'disk-lines-per-file': KeyInfo(default=1000),
                'disk-stop-on-full': KeyInfo(default=False),
                'email-start-tls': KeyInfo(default=False),
                'email-to': KeyInfo(default=''),
                'memory-lines': KeyInfo(default=1000),
                'memory-stop-on-full': KeyInfo(default=False),
                'name': KeyInfo(),
                'remember': KeyInfo(default=True),
                'remote': KeyInfo(default='0.0.0.0'),
                'remote-port': KeyInfo(default=514),
                'src-address': KeyInfo(default='0.0.0.0'),
                'syslog-facility': KeyInfo(default='daemon'),
                'syslog-severity': KeyInfo(default='auto'),
                'syslog-time-format': KeyInfo(default='bsd-syslog'),
                'target': KeyInfo(required=True),
            },
        ),
    ),
    ('user', 'group'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'name': KeyInfo(),
                'policy': KeyInfo(),
                'skin': KeyInfo(default='default'),
            },
        ),
    ),
    ('caps-man', 'manager'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'ca-certificate': KeyInfo(default='none'),
                'certificate': KeyInfo(default='none'),
                'enabled': KeyInfo(default=False),
                'package-path': KeyInfo(default=''),
                'require-peer-certificate': KeyInfo(default=False),
                'upgrade-policy': KeyInfo(default='none'),
            },
        ),
    ),
    ('ip', 'firewall', 'service-port'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=False),
                'name': KeyInfo(),
                'ports': KeyInfo(),
                'sip-direct-media': KeyInfo(),
                'sip-timeout': KeyInfo(),
            },
        ),
    ),
    ('ip', 'firewall', 'layer7-protocol'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'name': KeyInfo(),
                'regexp': KeyInfo(),
            },
        ),
    ),
    ('ip', 'hotspot', 'service-port'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=False),
                'name': KeyInfo(),
                'ports': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'policy'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'action': KeyInfo(default='encrypt'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'dst-address': KeyInfo(),
                'dst-port': KeyInfo(default='any'),
                'group': KeyInfo(can_disable=True, remove_value='default'),
                'ipsec-protocols': KeyInfo(default='esp'),
                'level': KeyInfo(default='require'),
                'peer': KeyInfo(),
                'proposal': KeyInfo(default='default'),
                'protocol': KeyInfo(default='all'),
                'src-address': KeyInfo(),
                'src-port': KeyInfo(default='any'),
                'template': KeyInfo(can_disable=True, remove_value=False),
                # the tepmlate field can't really be changed once the item is created. This config captures the behavior best as it can
                # i.e. tepmplate=yes is shown, tepmlate=no is hidden
                'tunnel': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'service'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'address': KeyInfo(),
                'certificate': KeyInfo(),
                'disabled': KeyInfo(default=False),
                'name': KeyInfo(),
                'port': KeyInfo(),
                'tls-version': KeyInfo(),
            },
        ),
    ),
    ('system', 'logging'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'action': KeyInfo(default='memory'),
                'disabled': KeyInfo(default=False),
                'prefix': KeyInfo(default=''),
                'topics': KeyInfo(default=''),
            },
        ),
    ),
    ('system', 'resource', 'irq'): APIData(
        unversioned=VersionedAPIData(
            has_identifier=True,
            fields={
                'cpu': KeyInfo(),
            },
        ),
    ),
    ('system', 'scheduler'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interval': KeyInfo(default='0s'),
                'name': KeyInfo(),
                'on-event': KeyInfo(default=''),
                'policy': KeyInfo(default='ftp,reboot,read,write,policy,test,password,sniff,sensitive,romon'),
                'start-date': KeyInfo(),
                'start-time': KeyInfo(),
            },
        ),
    ),
    ('system', 'script'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dont-require-permissions': KeyInfo(default=False),
                'name': KeyInfo(),
                'owner': KeyInfo(),
                'policy': KeyInfo(default='ftp,reboot,read,write,policy,test,password,sniff,sensitive,romon'),
                'source': KeyInfo(default=''),
            },
        ),
    ),
}
