from collections import OrderedDict
from typing import Type, Tuple, Mapping, Callable

import sqlalchemy
from graphene import Field
from graphene import InputObjectType, Dynamic, Argument, Mutation, ID
from graphene.relay import Connection, Node
from graphene.types.objecttype import ObjectType, ObjectTypeOptions
from graphene.types.structures import Structure
from graphene.types.utils import yank_fields_from_attrs
from graphene.utils.orderedtype import OrderedType
from graphene.utils.str_converters import to_snake_case
from graphql import ResolveInfo
from sqlalchemy.ext.declarative import DeclarativeMeta
from sqlalchemy.ext.hybrid import hybrid_property
from sqlalchemy.inspection import inspect as sqlalchemyinspect
from sqlalchemy.orm import (ColumnProperty, CompositeProperty,
                            RelationshipProperty)
from sqlalchemy.orm.exc import NoResultFound

from .converter import (
    convert_sqlalchemy_column,
    convert_sqlalchemy_composite,
    convert_sqlalchemy_hybrid_method,
    convert_sqlalchemy_relationship,
)
from .enums import (
    enum_for_field,
    sort_argument_for_object_type,
    sort_enum_for_object_type,
)
from .fields import SQLAlchemyFilteredConnectionField
from .fields import default_connection_field_factory
from .interfaces import SQLAlchemyInterface
from .registry import Registry, get_global_registry
from .utils import (
    get_query,
    is_mapped_class,
    is_mapped_instance,
    get_session,
    pluralize_name,
)


class ORMField(OrderedType):
    def __init__(
            self,
            model_attr=None,
            type=None,
            required=None,
            description=None,
            deprecation_reason=None,
            _creation_counter=None,
            **field_kwargs
    ):
        """
        Use this to override fields automatically generated by SQLAlchemyObjectType.
        Unless specified, options will default to SQLAlchemyObjectType usual behavior
        for the given SQLAlchemy model property.

        Usage:
            class MyModel(Base):
                id = Column(Integer(), primary_key=True)
                name = Column(String)

            class MyType(SQLAlchemyObjectType):
                class Meta:
                    model = MyModel

                id = ORMField(type=graphene.Int)
                name = ORMField(required=True)

        -> MyType.id will be of type Int (vs ID).
        -> MyType.name will be of type NonNull(String) (vs String).

        :param str model_attr:
            Name of the SQLAlchemy model attribute used to resolve this field.
            Default to the name of the attribute referencing the ORMField.
        :param type:
            Default to the type mapping in converter.py.
        :param str description:
            Default to the `doc` attribute of the SQLAlchemy column property.
        :param bool required:
            Default to the opposite of the `nullable` attribute of the SQLAlchemy column property.
        :param str description:
            Same behavior as in graphene.Field. Defaults to None.
        :param str deprecation_reason:
            Same behavior as in graphene.Field. Defaults to None.
        :param int _creation_counter:
            Same behavior as in graphene.Field.
        """
        super(ORMField, self).__init__(_creation_counter=_creation_counter)
        # The is only useful for documentation and auto-completion
        common_kwargs = {
            'model_attr': model_attr,
            'type': type,
            'required': required,
            'description': description,
            'deprecation_reason': deprecation_reason,
        }
        common_kwargs = {kwarg: value for kwarg, value in common_kwargs.items() if value is not None}
        self.kwargs = field_kwargs
        self.kwargs.update(common_kwargs)


def construct_fields(
        obj_type,
        model,
        registry,
        only_fields,
        exclude_fields,
        connection_field_factory,
        register_orm_field: bool = True,
):
    """
    Construct all the fields for a SQLAlchemyObjectType.
    The main steps are:
      - Gather all the relevant attributes from the SQLAlchemy model
      - Gather all the ORM fields defined on the type
      - Merge in overrides and build up all the fields

    :param SQLAlchemyObjectType obj_type:
    :param model: the SQLAlchemy model
    :param Registry registry:
    :param tuple[string] only_fields:
    :param tuple[string] exclude_fields:
    :param function connection_field_factory:
    :rtype: OrderedDict[str, graphene.Field]
    """
    inspected_model = sqlalchemyinspect(model)
    # Gather all the relevant attributes from the SQLAlchemy model in order
    all_model_attrs = OrderedDict(
        inspected_model.column_attrs.items() +
        inspected_model.composites.items() +
        [(name, item) for name, item in inspected_model.all_orm_descriptors.items()
         if isinstance(item, hybrid_property)] +
        inspected_model.relationships.items()
    )

    # Filter out excluded fields
    auto_orm_field_names = []
    for attr_name, attr in all_model_attrs.items():
        if (only_fields and attr_name not in only_fields) or (attr_name in exclude_fields):
            continue
        auto_orm_field_names.append(attr_name)

    # Gather all the ORM fields defined on the type
    custom_orm_fields_items = [
        (attn_name, attr)
        for base in reversed(obj_type.__mro__)
        for attn_name, attr in base.__dict__.items()
        if isinstance(attr, ORMField)
    ]
    custom_orm_fields_items = sorted(custom_orm_fields_items, key=lambda item: item[1])

    # Set the model_attr if not set
    for orm_field_name, orm_field in custom_orm_fields_items:
        attr_name = orm_field.kwargs.get('model_attr', orm_field_name)
        if attr_name not in all_model_attrs:
            raise ValueError((
                                 "Cannot map ORMField to a model attribute.\n"
                                 "Field: '{}.{}'"
                             ).format(obj_type.__name__, orm_field_name, ))
        orm_field.kwargs['model_attr'] = attr_name

    # Merge automatic fields with custom ORM fields
    orm_fields = OrderedDict(custom_orm_fields_items)
    for orm_field_name in auto_orm_field_names:
        if orm_field_name in orm_fields:
            continue
        orm_fields[orm_field_name] = ORMField(model_attr=orm_field_name)

    # Build all the field dictionary
    fields = OrderedDict()
    for orm_field_name, orm_field in orm_fields.items():
        attr_name = orm_field.kwargs.pop('model_attr')
        attr = all_model_attrs[attr_name]

        if isinstance(attr, ColumnProperty):
            field = convert_sqlalchemy_column(attr, registry, **orm_field.kwargs)
        elif isinstance(attr, RelationshipProperty):
            field = convert_sqlalchemy_relationship(attr, registry, connection_field_factory, **orm_field.kwargs)
        elif isinstance(attr, CompositeProperty):
            if attr_name != orm_field_name or orm_field.kwargs:
                # TODO Add a way to override composite property fields
                raise ValueError(
                    "ORMField kwargs for composite fields must be empty. "
                    "Field: {}.{}".format(obj_type.__name__, orm_field_name))
            field = convert_sqlalchemy_composite(attr, registry)
        elif isinstance(attr, hybrid_property):
            field = convert_sqlalchemy_hybrid_method(attr, attr_name, **orm_field.kwargs)
        else:
            raise Exception('Property type is not supported')  # Should never happen
        if register_orm_field:
            registry.register_orm_field(obj_type, orm_field_name, attr)
        fields[orm_field_name] = field

    return fields


class SQLAlchemyObjectTypeOptions(ObjectTypeOptions):
    model = None  # type: sqlalchemy.Model
    registry = None  # type: sqlalchemy.Registry
    connection = None  # type: sqlalchemy.Type[sqlalchemy.Connection]
    id = None  # type: str


class SQLAlchemyObjectType(ObjectType):
    @classmethod
    def __init_subclass_with_meta__(
            cls,
            model=None,
            registry=None,
            skip_registry=False,
            only_fields=(),
            exclude_fields=(),
            connection=None,
            connection_class=None,
            use_connection=None,
            interfaces=(),
            id=None,
            connection_field_factory=default_connection_field_factory,
            _meta=None,
            **options
    ):
        assert is_mapped_class(model), (
            "You need to pass a valid SQLAlchemy Model in " '{}.Meta, received "{}".'
        ).format(cls.__name__, model)

        if not registry:
            registry = get_global_registry()

        assert isinstance(registry, Registry), (
            "The attribute registry in {} needs to be an instance of "
            'Registry, received "{}".'
        ).format(cls.__name__, registry)

        if only_fields and exclude_fields:
            raise ValueError("The options 'only_fields' and 'exclude_fields' cannot be both set on the same type.")

        sqla_fields = yank_fields_from_attrs(
            construct_fields(
                obj_type=cls,
                model=model,
                registry=registry,
                only_fields=only_fields,
                exclude_fields=exclude_fields,
                connection_field_factory=connection_field_factory,
            ),
            _as=Field,
            sort=False,
        )

        if use_connection is None and interfaces:
            use_connection = any(
                (issubclass(interface, Node) for interface in interfaces)
            )

        if use_connection and not connection:
            # We create the connection automatically
            if not connection_class:
                connection_class = Connection

            connection = connection_class.create_type(
                "{}Connection".format(cls.__name__), node=cls
            )

        if connection is not None:
            assert issubclass(connection, Connection), (
                "The connection must be a Connection. Received {}"
            ).format(connection.__name__)

        if not _meta:
            _meta = SQLAlchemyObjectTypeOptions(cls)

        _meta.model = model
        _meta.registry = registry

        if _meta.fields:
            _meta.fields.update(sqla_fields)
        else:
            _meta.fields = sqla_fields

        _meta.connection = connection
        _meta.id = id or "id"

        super(SQLAlchemyObjectType, cls).__init_subclass_with_meta__(
            _meta=_meta, interfaces=interfaces, **options
        )

        if not skip_registry:
            registry.register(cls)

    @classmethod
    def is_type_of(cls, root, info):
        if isinstance(root, cls):
            return True
        if not is_mapped_instance(root):
            raise Exception(('Received incompatible instance "{}".').format(root))
        return isinstance(root, cls._meta.model)

    @classmethod
    def get_query(cls, info):
        model = cls._meta.model
        return get_query(model, info.context)

    @classmethod
    def get_node(cls, info, id):
        try:
            return cls.get_query(info).get(id)
        except NoResultFound:
            return None

    def resolve_id(self, info):
        # graphene_type = info.parent_type.graphene_type
        keys = self.__mapper__.primary_key_from_instance(self)
        return tuple(keys) if len(keys) > 1 else keys[0]

    @classmethod
    def enum_for_field(cls, field_name):
        return enum_for_field(cls, field_name)

    sort_enum = classmethod(sort_enum_for_object_type)

    sort_argument = classmethod(sort_argument_for_object_type)


class SQLAlchemyInputObjectType(InputObjectType):
    @classmethod
    def __init_subclass_with_meta__(
            cls,
            model=None,
            registry=None,
            skip_registry=False,
            only_fields=(),
            exclude_fields=(),
            connection=None,
            connection_class=None,
            use_connection=None,
            interfaces=(),
            id=None,
            connection_field_factory=default_connection_field_factory,
            _meta=None,
            **options,
    ):
        autoexclude = []

        # always pull ids out to a separate argument
        for col in sqlalchemy.inspect(model).columns:
            if (col.primary_key and col.autoincrement) or (
                    isinstance(col.type, sqlalchemy.types.TIMESTAMP)
                    and col.server_default is not None
            ):
                autoexclude.append(col.name)

        if not registry:
            registry = get_global_registry()
        sqla_fields = yank_fields_from_attrs(
            construct_fields(
                cls,
                model,
                registry,
                only_fields,
                exclude_fields + tuple(autoexclude),
                connection_field_factory,
            ),
            _as=Field,
        )
        # create accessor for model to be retrieved for querying
        cls.sqla_model = model
        if use_connection is None and interfaces:
            use_connection = any(
                (issubclass(interface, Node) for interface in interfaces)
            )

        if use_connection and not connection:
            # We create the connection automatically
            if not connection_class:
                connection_class = Connection

            connection = connection_class.create_type(
                "{}Connection".format(cls.__name__), node=cls
            )

        if connection is not None:
            assert issubclass(connection, Connection), (
                "The connection must be a Connection. Received {}"
            ).format(connection.__name__)

        for key, value in sqla_fields.items():
            if not (isinstance(value, Dynamic) or hasattr(cls, key)):
                setattr(cls, key, value)

        super(SQLAlchemyInputObjectType, cls).__init_subclass_with_meta__(**options)


class SQLAlchemyAutoSchemaFactory(ObjectType):
    @staticmethod
    def set_fields_and_attrs(
            klazz: Type[ObjectType],
            node_model: Type[SQLAlchemyInterface],
            field_dict: Mapping[str, Field],
    ):
        _name = to_snake_case(node_model.__name__)
        field_dict[
            f"all_{(pluralize_name(_name))}"
        ] = SQLAlchemyFilteredConnectionField(node_model)
        field_dict[_name] = node_model.Field()
        setattr(klazz, _name, node_model.Field())
        setattr(
            klazz,
            "all_{}".format(pluralize_name(_name)),
            SQLAlchemyFilteredConnectionField(node_model),
        )

    @classmethod
    def __init_subclass_with_meta__(
            cls,
            interfaces: Tuple[Type[SQLAlchemyInterface]] = (),
            models: Tuple[Type[DeclarativeMeta]] = (),
            excluded_models: Tuple[Type[DeclarativeMeta]] = (),
            node_interface: Type[Node] = Node,
            default_resolver: ResolveInfo = None,
            _meta=None,
            **options,
    ):
        if not _meta:
            _meta = ObjectTypeOptions(cls)

        fields = OrderedDict()

        for interface in interfaces:
            if issubclass(interface, SQLAlchemyInterface):
                SQLAlchemyAutoSchemaFactory.set_fields_and_attrs(cls, interface, fields)
        for model in excluded_models:
            if model in models:
                models = (
                        models[: models.index(model)] + models[models.index(model) + 1:]
                )
        possible_types = ()
        for model in models:
            model_name = model.__name__
            _model_name = to_snake_case(model.__name__)

            if hasattr(cls, _model_name):
                continue
            if hasattr(cls, "all_{}".format(pluralize_name(_model_name))):
                continue
            for iface in interfaces:
                if issubclass(model, iface._meta.model):
                    model_interface = iface
                    break
            else:
                model_interface = node_interface

            _node_class = type(
                model_name,
                (SQLAlchemyObjectType,),
                {
                    "Meta": {
                        "model": model,
                        "interfaces": (model_interface,),
                        "only_fields": [],
                    }
                },
            )
            fields[
                "all_{}".format(pluralize_name(_model_name))
            ] = SQLAlchemyFilteredConnectionField(_node_class)
            setattr(
                cls,
                "all_{}".format(pluralize_name(_model_name)),
                SQLAlchemyFilteredConnectionField(_node_class),
            )
            fields[_model_name] = model_interface.Field(_node_class)
            setattr(cls, _model_name, model_interface.Field(_node_class))
            possible_types += (_node_class,)
        if _meta.fields:
            _meta.fields.update(fields)
        else:
            _meta.fields = fields
        _meta.schema_types = possible_types

        super(SQLAlchemyAutoSchemaFactory, cls).__init_subclass_with_meta__(
            _meta=_meta, default_resolver=default_resolver, **options
        )

    @classmethod
    def resolve_with_filters(
            cls, info: ResolveInfo, model: Type[SQLAlchemyObjectType], **kwargs
    ):
        query = model.get_query(info)
        for filter_name, filter_value in kwargs.items():
            model_filter_column = getattr(model._meta.model, filter_name, None)
            if not model_filter_column:
                continue
            if isinstance(filter_value, SQLAlchemyInputObjectType):
                filter_model = filter_value.sqla_model
                q = SQLAlchemyFilteredConnectionField.get_query(
                    filter_model, info, sort=None, **kwargs
                )
                # noinspection PyArgumentList
                query = query.filter(model_filter_column == q.filter_by(**filter_value))
            else:
                query = query.filter(model_filter_column == filter_value)
        return query


class SQLAlchemyMutationOptions(ObjectTypeOptions):
    model: DeclarativeMeta = None
    create: bool = False
    delete: bool = False
    arguments: Mapping[str, Argument] = None
    output: Type[ObjectType] = None
    resolver: Callable = None


class SQLAlchemyMutation(Mutation):
    @classmethod
    def __init_subclass_with_meta__(
            cls,
            model=None,
            create=False,
            delete=False,
            registry=None,
            arguments=None,
            only_fields=(),
            structure: Type[Structure] = None,
            exclude_fields=(),
            **options,
    ):
        meta = SQLAlchemyMutationOptions(cls)
        meta.create = create
        meta.model = model
        meta.delete = delete

        if arguments is None and not hasattr(cls, "Arguments"):
            arguments = {}
            # don't include id argument on create
            if not meta.create:
                arguments["id"] = ID(required=True)

            # don't include input argument on delete
            if not meta.delete:
                inputMeta = type(
                    "Meta",
                    (object,),
                    {
                        "model": model,
                        "exclude_fields": exclude_fields,
                        "only_fields": only_fields,
                    },
                )
                inputType = type(
                    cls.__name__ + "Input",
                    (SQLAlchemyInputObjectType,),
                    {"Meta": inputMeta},
                )
                arguments = {"input": inputType(required=True)}
        if not registry:
            registry = get_global_registry()
        output_type: ObjectType = registry.get_type_for_model(model)
        if structure:
            output_type = structure(output_type)
        super(SQLAlchemyMutation, cls).__init_subclass_with_meta__(
            _meta=meta, output=output_type, arguments=arguments, **options
        )

    @classmethod
    def mutate(cls, root, info, **kwargs):
        session = get_session(info.context)
        with session.no_autoflush:
            meta = cls._meta

            if meta.create:
                model = meta.model(**kwargs["input"])
                session.add(model)
            else:
                model = (
                    session.query(meta.model)
                        .filter(meta.model.id == kwargs["id"])
                        .first()
                )
            if meta.delete:
                session.delete(model)
            else:

                def setModelAttributes(model, attrs):
                    relationships = model.__mapper__.relationships
                    for key, value in attrs.items():
                        if key in relationships:
                            if getattr(model, key) is None:
                                # instantiate class of the same type as
                                # the relationship target
                                setattr(model, key, relationships[key].mapper.entity())
                            setModelAttributes(getattr(model, key), value)
                        else:
                            setattr(model, key, value)

                setModelAttributes(model, kwargs["input"])
            session.flush()  # session.commit() now throws session state exception: 'already committed'

            return model

    @classmethod
    def Field(cls, *args, **kwargs):
        return Field(
            cls._meta.output, args=cls._meta.arguments, resolver=cls._meta.resolver
        )
