#!/usr/bin/python
from __future__ import absolute_import, division, print_function

# Copyright: (c) 2022 Fortinet
# GNU General Public License v3.0+ (see LICENSE or https://www.gnu.org/licenses/gpl-3.0.txt)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

__metaclass__ = type

ANSIBLE_METADATA = {
    "status": ["preview"],
    "supported_by": "community",
    "metadata_version": "1.1",
}

DOCUMENTATION = """
---
module: fortios_vpn_ssl_web_portal
short_description: Portal in Fortinet's FortiOS and FortiGate.
description:
    - This module is able to configure a FortiGate or FortiOS (FOS) device by allowing the
      user to set and modify vpn_ssl_web feature and portal category.
      Examples include all parameters and values need to be adjusted to datasources before usage.
      Tested with FOS v6.0.0
version_added: "2.0.0"
author:
    - Link Zheng (@chillancezen)
    - Jie Xue (@JieX19)
    - Hongbin Lu (@fgtdev-hblu)
    - Frank Shen (@frankshen01)
    - Miguel Angel Munoz (@mamunozgonzalez)
    - Nicolas Thomas (@thomnico)
notes:
    - Legacy fortiosapi has been deprecated, httpapi is the preferred way to run playbooks

requirements:
    - ansible>=2.9
options:
    access_token:
        description:
            - Token-based authentication.
              Generated from GUI of Fortigate.
        type: str
        required: false
    enable_log:
        description:
            - Enable/Disable logging for task.
        type: bool
        required: false
        default: false
    vdom:
        description:
            - Virtual domain, among those defined previously. A vdom is a
              virtual instance of the FortiGate that can be configured and
              used as a different unit.
        type: str
        default: root
    member_path:
        type: str
        description:
            - Member attribute path to operate on.
            - Delimited by a slash character if there are more than one attribute.
            - Parameter marked with member_path is legitimate for doing member operation.
    member_state:
        type: str
        description:
            - Add or delete a member under specified attribute path.
            - When member_state is specified, the state option is ignored.
        choices:
            - 'present'
            - 'absent'

    state:
        description:
            - Indicates whether to create or remove the object.
        type: str
        required: true
        choices:
            - 'present'
            - 'absent'
    vpn_ssl_web_portal:
        description:
            - Portal.
        default: null
        type: dict
        suboptions:
            allow_user_access:
                description:
                    - Allow user access to SSL-VPN applications.
                type: list
                elements: str
                choices:
                    - 'web'
                    - 'ftp'
                    - 'smb'
                    - 'sftp'
                    - 'telnet'
                    - 'ssh'
                    - 'vnc'
                    - 'rdp'
                    - 'ping'
                    - 'citrix'
                    - 'portforward'
            auto_connect:
                description:
                    - Enable/disable automatic connect by client when system is up.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            bookmark_group:
                description:
                    - Portal bookmark group.
                type: list
                elements: dict
                suboptions:
                    bookmarks:
                        description:
                            - Bookmark table.
                        type: list
                        elements: dict
                        suboptions:
                            additional_params:
                                description:
                                    - Additional parameters.
                                type: str
                            apptype:
                                description:
                                    - Application type.
                                type: str
                                choices:
                                    - 'ftp'
                                    - 'rdp'
                                    - 'sftp'
                                    - 'smb'
                                    - 'ssh'
                                    - 'telnet'
                                    - 'vnc'
                                    - 'web'
                                    - 'citrix'
                                    - 'portforward'
                            color_depth:
                                description:
                                    - Color depth per pixel.
                                type: str
                                choices:
                                    - '32'
                                    - '16'
                                    - '8'
                            description:
                                description:
                                    - Description.
                                type: str
                            domain:
                                description:
                                    - Login domain.
                                type: str
                            folder:
                                description:
                                    - Network shared file folder parameter.
                                type: str
                            form_data:
                                description:
                                    - Form data.
                                type: list
                                elements: dict
                                suboptions:
                                    name:
                                        description:
                                            - Name.
                                        required: true
                                        type: str
                                    value:
                                        description:
                                            - Value.
                                        type: str
                            height:
                                description:
                                    - Screen height (range from 0 - 65535).
                                type: int
                            host:
                                description:
                                    - Host name/IP parameter.
                                type: str
                            keyboard_layout:
                                description:
                                    - Keyboard layout.
                                type: str
                                choices:
                                    - 'ar-101'
                                    - 'ar-102'
                                    - 'ar-102-azerty'
                                    - 'can-mul'
                                    - 'cz'
                                    - 'cz-qwerty'
                                    - 'cz-pr'
                                    - 'da'
                                    - 'nl'
                                    - 'de'
                                    - 'de-ch'
                                    - 'de-ibm'
                                    - 'en-uk'
                                    - 'en-uk-ext'
                                    - 'en-us'
                                    - 'en-us-dvorak'
                                    - 'es'
                                    - 'es-var'
                                    - 'fi'
                                    - 'fi-sami'
                                    - 'fr'
                                    - 'fr-apple'
                                    - 'fr-ca'
                                    - 'fr-ch'
                                    - 'fr-be'
                                    - 'hr'
                                    - 'hu'
                                    - 'hu-101'
                                    - 'it'
                                    - 'it-142'
                                    - 'ja'
                                    - 'ko'
                                    - 'lt'
                                    - 'lt-ibm'
                                    - 'lt-std'
                                    - 'lav-std'
                                    - 'lav-leg'
                                    - 'mk'
                                    - 'mk-std'
                                    - 'no'
                                    - 'no-sami'
                                    - 'pol-214'
                                    - 'pol-pr'
                                    - 'pt'
                                    - 'pt-br'
                                    - 'pt-br-abnt2'
                                    - 'ru'
                                    - 'ru-mne'
                                    - 'ru-t'
                                    - 'sl'
                                    - 'sv'
                                    - 'sv-sami'
                                    - 'tuk'
                                    - 'tur-f'
                                    - 'tur-q'
                                    - 'zh-sym-sg-us'
                                    - 'zh-sym-us'
                                    - 'zh-tr-hk'
                                    - 'zh-tr-mo'
                                    - 'zh-tr-us'
                            listening_port:
                                description:
                                    - Listening port (0 - 65535).
                                type: int
                            load_balancing_info:
                                description:
                                    - The load balancing information or cookie which should be provided to the connection broker.
                                type: str
                            logon_password:
                                description:
                                    - Logon password.
                                type: str
                            logon_user:
                                description:
                                    - Logon user.
                                type: str
                            name:
                                description:
                                    - Bookmark name.
                                required: true
                                type: str
                            port:
                                description:
                                    - Remote port.
                                type: int
                            preconnection_blob:
                                description:
                                    - An arbitrary string which identifies the RDP source.
                                type: str
                            preconnection_id:
                                description:
                                    - The numeric ID of the RDP source (0-4294967295).
                                type: int
                            remote_port:
                                description:
                                    - Remote port (0 - 65535).
                                type: int
                            restricted_admin:
                                description:
                                    - Enable/disable restricted admin mode for RDP.
                                type: str
                                choices:
                                    - 'enable'
                                    - 'disable'
                            security:
                                description:
                                    - Security mode for RDP connection.
                                type: str
                                choices:
                                    - 'any'
                                    - 'rdp'
                                    - 'nla'
                                    - 'tls'
                            send_preconnection_id:
                                description:
                                    - Enable/disable sending of preconnection ID.
                                type: str
                                choices:
                                    - 'enable'
                                    - 'disable'
                            server_layout:
                                description:
                                    - Server side keyboard layout.
                                type: str
                                choices:
                                    - 'de-de-qwertz'
                                    - 'en-gb-qwerty'
                                    - 'en-us-qwerty'
                                    - 'es-es-qwerty'
                                    - 'fr-ca-qwerty'
                                    - 'fr-fr-azerty'
                                    - 'fr-ch-qwertz'
                                    - 'it-it-qwerty'
                                    - 'ja-jp-qwerty'
                                    - 'pt-br-qwerty'
                                    - 'sv-se-qwerty'
                                    - 'tr-tr-qwerty'
                                    - 'failsafe'
                            show_status_window:
                                description:
                                    - Enable/disable showing of status window.
                                type: str
                                choices:
                                    - 'enable'
                                    - 'disable'
                            sso:
                                description:
                                    - Single sign-on.
                                type: str
                                choices:
                                    - 'disable'
                                    - 'static'
                                    - 'auto'
                            sso_credential:
                                description:
                                    - Single sign-on credentials.
                                type: str
                                choices:
                                    - 'sslvpn-login'
                                    - 'alternative'
                            sso_credential_sent_once:
                                description:
                                    - Single sign-on credentials are only sent once to remote server.
                                type: str
                                choices:
                                    - 'enable'
                                    - 'disable'
                            sso_password:
                                description:
                                    - SSO password.
                                type: str
                            sso_username:
                                description:
                                    - SSO user name.
                                type: str
                            url:
                                description:
                                    - URL parameter.
                                type: str
                            vnc_keyboard_layout:
                                description:
                                    - Keyboard layout.
                                type: str
                                choices:
                                    - 'default'
                                    - 'da'
                                    - 'nl'
                                    - 'en-uk'
                                    - 'en-uk-ext'
                                    - 'fi'
                                    - 'fr'
                                    - 'fr-be'
                                    - 'fr-ca-mul'
                                    - 'de'
                                    - 'de-ch'
                                    - 'it'
                                    - 'it-142'
                                    - 'pt'
                                    - 'pt-br-abnt2'
                                    - 'no'
                                    - 'gd'
                                    - 'es'
                                    - 'sv'
                                    - 'us-intl'
                            width:
                                description:
                                    - Screen width (range from 0 - 65535).
                                type: int
                    name:
                        description:
                            - Bookmark group name.
                        required: true
                        type: str
            client_src_range:
                description:
                    - Allow client to add source range for the tunnel traffic.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            clipboard:
                description:
                    - Enable to support RDP/VPC clipboard functionality.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            custom_lang:
                description:
                    - Change the web portal display language. Overrides config system global set language. You can use config system custom-language and
                       execute system custom-language to add custom language files. Source system.custom-language.name.
                type: str
            customize_forticlient_download_url:
                description:
                    - Enable support of customized download URL for FortiClient.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            default_window_height:
                description:
                    - Screen height (range from 0 - 65535).
                type: int
            default_window_width:
                description:
                    - Screen width (range from 0 - 65535).
                type: int
            dhcp_ip_overlap:
                description:
                    - Configure overlapping DHCP IP allocation assignment.
                type: str
                choices:
                    - 'use-new'
                    - 'use-old'
            dhcp_ra_giaddr:
                description:
                    - Relay agent gateway IP address to use in the giaddr field of DHCP requests.
                type: str
            dhcp6_ra_linkaddr:
                description:
                    - Relay agent IPv6 link address to use in DHCP6 requests.
                type: str
            display_bookmark:
                description:
                    - Enable to display the web portal bookmark widget.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            display_connection_tools:
                description:
                    - Enable to display the web portal connection tools widget.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            display_history:
                description:
                    - Enable to display the web portal user login history widget.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            display_status:
                description:
                    - Enable to display the web portal status widget.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            dns_server1:
                description:
                    - IPv4 DNS server 1.
                type: str
            dns_server2:
                description:
                    - IPv4 DNS server 2.
                type: str
            dns_suffix:
                description:
                    - DNS suffix.
                type: str
            exclusive_routing:
                description:
                    - Enable/disable all traffic go through tunnel only.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            forticlient_download:
                description:
                    - Enable/disable download option for FortiClient.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            forticlient_download_method:
                description:
                    - FortiClient download method.
                type: str
                choices:
                    - 'direct'
                    - 'ssl-vpn'
            heading:
                description:
                    - Web portal heading message.
                type: str
            hide_sso_credential:
                description:
                    - Enable to prevent SSO credential being sent to client.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            host_check:
                description:
                    - Type of host checking performed on endpoints.
                type: str
                choices:
                    - 'none'
                    - 'av'
                    - 'fw'
                    - 'av-fw'
                    - 'custom'
            host_check_interval:
                description:
                    - Periodic host check interval. Value of 0 means disabled and host checking only happens when the endpoint connects.
                type: int
            host_check_policy:
                description:
                    - One or more policies to require the endpoint to have specific security software.
                type: list
                elements: dict
                suboptions:
                    name:
                        description:
                            - Host check software list name. Source vpn.ssl.web.host-check-software.name.
                        required: true
                        type: str
            ip_mode:
                description:
                    - Method by which users of this SSL-VPN tunnel obtain IP addresses.
                type: str
                choices:
                    - 'range'
                    - 'user-group'
                    - 'dhcp'
                    - 'no-ip'
            ip_pools:
                description:
                    - IPv4 firewall source address objects reserved for SSL-VPN tunnel mode clients.
                type: list
                elements: dict
                suboptions:
                    name:
                        description:
                            - Address name. Source firewall.address.name firewall.addrgrp.name.
                        required: true
                        type: str
            ipv6_dns_server1:
                description:
                    - IPv6 DNS server 1.
                type: str
            ipv6_dns_server2:
                description:
                    - IPv6 DNS server 2.
                type: str
            ipv6_exclusive_routing:
                description:
                    - Enable/disable all IPv6 traffic go through tunnel only.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            ipv6_pools:
                description:
                    - IPv6 firewall source address objects reserved for SSL-VPN tunnel mode clients.
                type: list
                elements: dict
                suboptions:
                    name:
                        description:
                            - Address name. Source firewall.address6.name firewall.addrgrp6.name.
                        required: true
                        type: str
            ipv6_service_restriction:
                description:
                    - Enable/disable IPv6 tunnel service restriction.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            ipv6_split_tunneling:
                description:
                    - Enable/disable IPv6 split tunneling.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            ipv6_split_tunneling_routing_address:
                description:
                    - IPv6 SSL-VPN tunnel mode firewall address objects that override firewall policy destination addresses to control split-tunneling access.
                type: list
                elements: dict
                suboptions:
                    name:
                        description:
                            - Address name. Source firewall.address6.name firewall.addrgrp6.name.
                        required: true
                        type: str
            ipv6_split_tunneling_routing_negate:
                description:
                    - Enable to negate IPv6 split tunneling routing address.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            ipv6_tunnel_mode:
                description:
                    - Enable/disable IPv6 SSL-VPN tunnel mode.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            ipv6_wins_server1:
                description:
                    - IPv6 WINS server 1.
                type: str
            ipv6_wins_server2:
                description:
                    - IPv6 WINS server 2.
                type: str
            keep_alive:
                description:
                    - Enable/disable automatic reconnect for FortiClient connections.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            landing_page:
                description:
                    - Landing page options.
                type: dict
                suboptions:
                    form_data:
                        description:
                            - Form data.
                        type: list
                        elements: dict
                        suboptions:
                            name:
                                description:
                                    - Name.
                                required: true
                                type: str
                            value:
                                description:
                                    - Value.
                                type: str
                    logout_url:
                        description:
                            - Landing page log out URL.
                        type: str
                    sso:
                        description:
                            - Single sign-on.
                        type: str
                        choices:
                            - 'disable'
                            - 'static'
                            - 'auto'
                    sso_credential:
                        description:
                            - Single sign-on credentials.
                        type: str
                        choices:
                            - 'sslvpn-login'
                            - 'alternative'
                    sso_password:
                        description:
                            - SSO password.
                        type: str
                    sso_username:
                        description:
                            - SSO user name.
                        type: str
                    url:
                        description:
                            - Landing page URL.
                        type: str
            landing_page_mode:
                description:
                    - Enable/disable SSL-VPN landing page mode.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            limit_user_logins:
                description:
                    - Enable to limit each user to one SSL-VPN session at a time.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            mac_addr_action:
                description:
                    - Client MAC address action.
                type: str
                choices:
                    - 'allow'
                    - 'deny'
            mac_addr_check:
                description:
                    - Enable/disable MAC address host checking.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            mac_addr_check_rule:
                description:
                    - Client MAC address check rule.
                type: list
                elements: dict
                suboptions:
                    mac_addr_list:
                        description:
                            - Client MAC address list.
                        type: list
                        elements: dict
                        suboptions:
                            addr:
                                description:
                                    - Client MAC address.
                                required: true
                                type: str
                    mac_addr_mask:
                        description:
                            - Client MAC address mask.
                        type: int
                    name:
                        description:
                            - Client MAC address check rule name.
                        required: true
                        type: str
            macos_forticlient_download_url:
                description:
                    - Download URL for Mac FortiClient.
                type: str
            name:
                description:
                    - Portal name.
                required: true
                type: str
            os_check:
                description:
                    - Enable to let the FortiGate decide action based on client OS.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            os_check_list:
                description:
                    - SSL-VPN OS checks.
                type: list
                elements: dict
                suboptions:
                    action:
                        description:
                            - OS check options.
                        type: str
                        choices:
                            - 'deny'
                            - 'allow'
                            - 'check-up-to-date'
                    latest_patch_level:
                        description:
                            - Latest OS patch level.
                        type: str
                    name:
                        description:
                            - Name.
                        required: true
                        type: str
                    tolerance:
                        description:
                            - OS patch level tolerance.
                        type: int
            prefer_ipv6_dns:
                description:
                    - Prefer to query IPv6 DNS server first if enabled.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            redir_url:
                description:
                    - Client login redirect URL.
                type: str
            rewrite_ip_uri_ui:
                description:
                    - Rewrite contents for URI contains IP and /ui/ .
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            save_password:
                description:
                    - Enable/disable FortiClient saving the user"s password.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            service_restriction:
                description:
                    - Enable/disable tunnel service restriction.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            skip_check_for_browser:
                description:
                    - Enable to skip host check for browser support.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            skip_check_for_unsupported_browser:
                description:
                    - Enable to skip host check if browser does not support it.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            skip_check_for_unsupported_os:
                description:
                    - Enable to skip host check if client OS does not support it.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            smb_max_version:
                description:
                    - SMB maximum client protocol version.
                type: str
                choices:
                    - 'smbv1'
                    - 'smbv2'
                    - 'smbv3'
            smb_min_version:
                description:
                    - SMB minimum client protocol version.
                type: str
                choices:
                    - 'smbv1'
                    - 'smbv2'
                    - 'smbv3'
            smb_ntlmv1_auth:
                description:
                    - Enable support of NTLMv1 for Samba authentication.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            smbv1:
                description:
                    - SMB version 1.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            split_dns:
                description:
                    - Split DNS for SSL-VPN.
                type: list
                elements: dict
                suboptions:
                    dns_server1:
                        description:
                            - DNS server 1.
                        type: str
                    dns_server2:
                        description:
                            - DNS server 2.
                        type: str
                    domains:
                        description:
                            - Split DNS domains used for SSL-VPN clients separated by comma.
                        type: str
                    id:
                        description:
                            - ID. see <a href='#notes'>Notes</a>.
                        required: true
                        type: int
                    ipv6_dns_server1:
                        description:
                            - IPv6 DNS server 1.
                        type: str
                    ipv6_dns_server2:
                        description:
                            - IPv6 DNS server 2.
                        type: str
            split_tunneling:
                description:
                    - Enable/disable IPv4 split tunneling.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            split_tunneling_routing_address:
                description:
                    - IPv4 SSL-VPN tunnel mode firewall address objects that override firewall policy destination addresses to control split-tunneling access.
                type: list
                elements: dict
                suboptions:
                    name:
                        description:
                            - Address name. Source firewall.address.name firewall.addrgrp.name.
                        required: true
                        type: str
            split_tunneling_routing_negate:
                description:
                    - Enable to negate split tunneling routing address.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            theme:
                description:
                    - Web portal color scheme.
                type: str
                choices:
                    - 'jade'
                    - 'neutrino'
                    - 'mariner'
                    - 'graphite'
                    - 'melongene'
                    - 'jet-stream'
                    - 'security-fabric'
                    - 'dark-matter'
                    - 'onyx'
                    - 'eclipse'
                    - 'blue'
                    - 'green'
                    - 'red'
            transform_backward_slashes:
                description:
                    - Transform backward slashes to forward slashes in URLs.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            tunnel_mode:
                description:
                    - Enable/disable IPv4 SSL-VPN tunnel mode.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            use_sdwan:
                description:
                    - Use SD-WAN rules to get output interface.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            user_bookmark:
                description:
                    - Enable to allow web portal users to create their own bookmarks.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            user_group_bookmark:
                description:
                    - Enable to allow web portal users to create bookmarks for all users in the same user group.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            web_mode:
                description:
                    - Enable/disable SSL-VPN web mode.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            windows_forticlient_download_url:
                description:
                    - Download URL for Windows FortiClient.
                type: str
            wins_server1:
                description:
                    - IPv4 WINS server 1.
                type: str
            wins_server2:
                description:
                    - IPv4 WINS server 1.
                type: str
"""

EXAMPLES = """
- hosts: fortigates
  collections:
    - fortinet.fortios
  connection: httpapi
  vars:
   vdom: "root"
   ansible_httpapi_use_ssl: yes
   ansible_httpapi_validate_certs: no
   ansible_httpapi_port: 443
  tasks:
  - name: Portal.
    fortios_vpn_ssl_web_portal:
      vdom:  "{{ vdom }}"
      state: "present"
      access_token: "<your_own_value>"
      vpn_ssl_web_portal:
        allow_user_access: "web"
        auto_connect: "enable"
        bookmark_group:
         -
            bookmarks:
             -
                additional_params: "<your_own_value>"
                apptype: "ftp"
                color_depth: "32"
                description: "<your_own_value>"
                domain: "<your_own_value>"
                folder: "<your_own_value>"
                form_data:
                 -
                    name: "default_name_14"
                    value: "<your_own_value>"
                height: "768"
                host: "myhostname"
                keyboard_layout: "ar-101"
                listening_port: "0"
                load_balancing_info: "<your_own_value>"
                logon_password: "<your_own_value>"
                logon_user: "<your_own_value>"
                name: "default_name_23"
                port: "0"
                preconnection_blob: "<your_own_value>"
                preconnection_id: "2147483648"
                remote_port: "0"
                restricted_admin: "enable"
                security: "any"
                send_preconnection_id: "enable"
                server_layout: "de-de-qwertz"
                show_status_window: "enable"
                sso: "disable"
                sso_credential: "sslvpn-login"
                sso_credential_sent_once: "enable"
                sso_password: "<your_own_value>"
                sso_username: "<your_own_value>"
                url: "myurl.com"
                vnc_keyboard_layout: "default"
                width: "1024"
            name: "default_name_41"
        client_src_range: "enable"
        clipboard: "enable"
        custom_lang: "<your_own_value> (source system.custom-language.name)"
        customize_forticlient_download_url: "enable"
        default_window_height: "768"
        default_window_width: "1024"
        dhcp_ip_overlap: "use-new"
        dhcp_ra_giaddr: "<your_own_value>"
        dhcp6_ra_linkaddr: "<your_own_value>"
        display_bookmark: "enable"
        display_connection_tools: "enable"
        display_history: "enable"
        display_status: "enable"
        dns_server1: "<your_own_value>"
        dns_server2: "<your_own_value>"
        dns_suffix: "<your_own_value>"
        exclusive_routing: "enable"
        forticlient_download: "enable"
        forticlient_download_method: "direct"
        heading: "<your_own_value>"
        hide_sso_credential: "enable"
        host_check: "none"
        host_check_interval: "0"
        host_check_policy:
         -
            name: "default_name_66 (source vpn.ssl.web.host-check-software.name)"
        ip_mode: "range"
        ip_pools:
         -
            name: "default_name_69 (source firewall.address.name firewall.addrgrp.name)"
        ipv6_dns_server1: "<your_own_value>"
        ipv6_dns_server2: "<your_own_value>"
        ipv6_exclusive_routing: "enable"
        ipv6_pools:
         -
            name: "default_name_74 (source firewall.address6.name firewall.addrgrp6.name)"
        ipv6_service_restriction: "enable"
        ipv6_split_tunneling: "enable"
        ipv6_split_tunneling_routing_address:
         -
            name: "default_name_78 (source firewall.address6.name firewall.addrgrp6.name)"
        ipv6_split_tunneling_routing_negate: "enable"
        ipv6_tunnel_mode: "enable"
        ipv6_wins_server1: "<your_own_value>"
        ipv6_wins_server2: "<your_own_value>"
        keep_alive: "enable"
        landing_page:
            form_data:
             -
                name: "default_name_86"
                value: "<your_own_value>"
            logout_url: "<your_own_value>"
            sso: "disable"
            sso_credential: "sslvpn-login"
            sso_password: "<your_own_value>"
            sso_username: "<your_own_value>"
            url: "myurl.com"
        landing_page_mode: "enable"
        limit_user_logins: "enable"
        mac_addr_action: "allow"
        mac_addr_check: "enable"
        mac_addr_check_rule:
         -
            mac_addr_list:
             -
                addr: "<your_own_value>"
            mac_addr_mask: "48"
            name: "default_name_102"
        macos_forticlient_download_url: "<your_own_value>"
        name: "default_name_104"
        os_check: "enable"
        os_check_list:
         -
            action: "deny"
            latest_patch_level: "<your_own_value>"
            name: "default_name_109"
            tolerance: "0"
        prefer_ipv6_dns: "enable"
        redir_url: "<your_own_value>"
        rewrite_ip_uri_ui: "enable"
        save_password: "enable"
        service_restriction: "enable"
        skip_check_for_browser: "enable"
        skip_check_for_unsupported_browser: "enable"
        skip_check_for_unsupported_os: "enable"
        smb_max_version: "smbv1"
        smb_min_version: "smbv1"
        smb_ntlmv1_auth: "enable"
        smbv1: "enable"
        split_dns:
         -
            dns_server1: "<your_own_value>"
            dns_server2: "<your_own_value>"
            domains: "<your_own_value>"
            id:  "127"
            ipv6_dns_server1: "<your_own_value>"
            ipv6_dns_server2: "<your_own_value>"
        split_tunneling: "enable"
        split_tunneling_routing_address:
         -
            name: "default_name_132 (source firewall.address.name firewall.addrgrp.name)"
        split_tunneling_routing_negate: "enable"
        theme: "jade"
        transform_backward_slashes: "enable"
        tunnel_mode: "enable"
        use_sdwan: "enable"
        user_bookmark: "enable"
        user_group_bookmark: "enable"
        web_mode: "enable"
        windows_forticlient_download_url: "<your_own_value>"
        wins_server1: "<your_own_value>"
        wins_server2: "<your_own_value>"

"""

RETURN = """
build:
  description: Build number of the fortigate image
  returned: always
  type: str
  sample: '1547'
http_method:
  description: Last method used to provision the content into FortiGate
  returned: always
  type: str
  sample: 'PUT'
http_status:
  description: Last result given by FortiGate on last operation applied
  returned: always
  type: str
  sample: "200"
mkey:
  description: Master key (id) used in the last call to FortiGate
  returned: success
  type: str
  sample: "id"
name:
  description: Name of the table used to fulfill the request
  returned: always
  type: str
  sample: "urlfilter"
path:
  description: Path of the table used to fulfill the request
  returned: always
  type: str
  sample: "webfilter"
revision:
  description: Internal revision number
  returned: always
  type: str
  sample: "17.0.2.10658"
serial:
  description: Serial number of the unit
  returned: always
  type: str
  sample: "FGVMEVYYQT3AB5352"
status:
  description: Indication of the operation's result
  returned: always
  type: str
  sample: "success"
vdom:
  description: Virtual domain used
  returned: always
  type: str
  sample: "root"
version:
  description: Version of the FortiGate
  returned: always
  type: str
  sample: "v5.6.3"

"""
from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils.connection import Connection
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    FortiOSHandler,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    check_legacy_fortiosapi,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    schema_to_module_spec,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    check_schema_versioning,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortimanager.common import (
    FAIL_SOCKET_MSG,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.data_post_processor import (
    remove_invalid_fields,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.comparison import (
    is_same_comparison,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.comparison import (
    serialize,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.comparison import (
    find_current_values,
)


def filter_vpn_ssl_web_portal_data(json):
    option_list = [
        "allow_user_access",
        "auto_connect",
        "bookmark_group",
        "client_src_range",
        "clipboard",
        "custom_lang",
        "customize_forticlient_download_url",
        "default_window_height",
        "default_window_width",
        "dhcp_ip_overlap",
        "dhcp_ra_giaddr",
        "dhcp6_ra_linkaddr",
        "display_bookmark",
        "display_connection_tools",
        "display_history",
        "display_status",
        "dns_server1",
        "dns_server2",
        "dns_suffix",
        "exclusive_routing",
        "forticlient_download",
        "forticlient_download_method",
        "heading",
        "hide_sso_credential",
        "host_check",
        "host_check_interval",
        "host_check_policy",
        "ip_mode",
        "ip_pools",
        "ipv6_dns_server1",
        "ipv6_dns_server2",
        "ipv6_exclusive_routing",
        "ipv6_pools",
        "ipv6_service_restriction",
        "ipv6_split_tunneling",
        "ipv6_split_tunneling_routing_address",
        "ipv6_split_tunneling_routing_negate",
        "ipv6_tunnel_mode",
        "ipv6_wins_server1",
        "ipv6_wins_server2",
        "keep_alive",
        "landing_page",
        "landing_page_mode",
        "limit_user_logins",
        "mac_addr_action",
        "mac_addr_check",
        "mac_addr_check_rule",
        "macos_forticlient_download_url",
        "name",
        "os_check",
        "os_check_list",
        "prefer_ipv6_dns",
        "redir_url",
        "rewrite_ip_uri_ui",
        "save_password",
        "service_restriction",
        "skip_check_for_browser",
        "skip_check_for_unsupported_browser",
        "skip_check_for_unsupported_os",
        "smb_max_version",
        "smb_min_version",
        "smb_ntlmv1_auth",
        "smbv1",
        "split_dns",
        "split_tunneling",
        "split_tunneling_routing_address",
        "split_tunneling_routing_negate",
        "theme",
        "transform_backward_slashes",
        "tunnel_mode",
        "use_sdwan",
        "user_bookmark",
        "user_group_bookmark",
        "web_mode",
        "windows_forticlient_download_url",
        "wins_server1",
        "wins_server2",
    ]

    json = remove_invalid_fields(json)
    dictionary = {}

    for attribute in option_list:
        if attribute in json and json[attribute] is not None:
            dictionary[attribute] = json[attribute]

    return dictionary


def flatten_single_path(data, path, index):
    if (
        not data
        or index == len(path)
        or path[index] not in data
        or not data[path[index]]
    ):
        return

    if index == len(path) - 1:
        data[path[index]] = " ".join(str(elem) for elem in data[path[index]])
    elif isinstance(data[path[index]], list):
        for value in data[path[index]]:
            flatten_single_path(value, path, index + 1)
    else:
        flatten_single_path(data[path[index]], path, index + 1)


def flatten_multilists_attributes(data):
    multilist_attrs = [
        ["allow_user_access"],
    ]

    for attr in multilist_attrs:
        flatten_single_path(data, attr, 0)

    return data


def underscore_to_hyphen(data):
    if isinstance(data, list):
        for i, elem in enumerate(data):
            data[i] = underscore_to_hyphen(elem)
    elif isinstance(data, dict):
        new_data = {}
        for k, v in data.items():
            new_data[k.replace("_", "-")] = underscore_to_hyphen(v)
        data = new_data

    return data


def vpn_ssl_web_portal(data, fos, check_mode=False):
    vdom = data["vdom"]

    state = data["state"]

    vpn_ssl_web_portal_data = data["vpn_ssl_web_portal"]
    vpn_ssl_web_portal_data = flatten_multilists_attributes(vpn_ssl_web_portal_data)
    filtered_data = underscore_to_hyphen(
        filter_vpn_ssl_web_portal_data(vpn_ssl_web_portal_data)
    )

    # check_mode starts from here
    if check_mode:
        diff = {
            "before": "",
            "after": filtered_data,
        }
        mkey = fos.get_mkey("vpn.ssl.web", "portal", filtered_data, vdom=vdom)
        current_data = fos.get("vpn.ssl.web", "portal", vdom=vdom, mkey=mkey)
        is_existed = (
            current_data
            and current_data.get("http_status") == 200
            and isinstance(current_data.get("results"), list)
            and len(current_data["results"]) > 0
        )

        # 2. if it exists and the state is 'present' then compare current settings with desired
        if state == "present" or state is True:
            if mkey is None:
                return False, True, filtered_data, diff

            # if mkey exists then compare each other
            # record exits and they're matched or not
            if is_existed:
                is_same = is_same_comparison(
                    serialize(current_data["results"][0]), serialize(filtered_data)
                )

                current_values = find_current_values(
                    current_data["results"][0], filtered_data
                )

                return (
                    False,
                    not is_same,
                    filtered_data,
                    {"before": current_values, "after": filtered_data},
                )

            # record does not exist
            return False, True, filtered_data, diff

        if state == "absent":
            if mkey is None:
                return (
                    False,
                    False,
                    filtered_data,
                    {"before": current_data["results"][0], "after": ""},
                )

            if is_existed:
                return (
                    False,
                    True,
                    filtered_data,
                    {"before": current_data["results"][0], "after": ""},
                )
            return False, False, filtered_data, {}

        return True, False, {"reason: ": "Must provide state parameter"}, {}

    if state == "present" or state is True:
        return fos.set("vpn.ssl.web", "portal", data=filtered_data, vdom=vdom)

    elif state == "absent":
        return fos.delete(
            "vpn.ssl.web", "portal", mkey=filtered_data["name"], vdom=vdom
        )
    else:
        fos._module.fail_json(msg="state must be present or absent!")


def is_successful_status(resp):
    return (
        "status" in resp
        and resp["status"] == "success"
        or "http_status" in resp
        and resp["http_status"] == 200
        or "http_method" in resp
        and resp["http_method"] == "DELETE"
        and resp["http_status"] == 404
    )


def fortios_vpn_ssl_web(data, fos, check_mode):
    fos.do_member_operation("vpn.ssl.web", "portal")
    if data["vpn_ssl_web_portal"]:
        resp = vpn_ssl_web_portal(data, fos, check_mode)
    else:
        fos._module.fail_json(msg="missing task body: %s" % ("vpn_ssl_web_portal"))
    if check_mode:
        return resp
    return (
        not is_successful_status(resp),
        is_successful_status(resp)
        and (resp["revision_changed"] if "revision_changed" in resp else True),
        resp,
        {},
    )


versioned_schema = {
    "type": "list",
    "elements": "dict",
    "children": {
        "name": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "required": True,
        },
        "tunnel_mode": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "ip_mode": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "range",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "user-group",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "dhcp",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": False,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": False,
                        "v7.0.4": False,
                        "v7.0.3": False,
                        "v7.0.2": False,
                        "v7.0.12": True,
                        "v7.0.1": False,
                        "v7.0.0": False,
                        "v6.4.4": False,
                        "v6.4.1": False,
                        "v6.4.0": False,
                        "v6.2.7": False,
                        "v6.2.5": False,
                        "v6.2.3": False,
                        "v6.2.0": False,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "no-ip",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": False,
                        "v7.2.1": False,
                        "v7.2.0": False,
                        "v7.0.8": False,
                        "v7.0.7": False,
                        "v7.0.6": False,
                        "v7.0.5": False,
                        "v7.0.4": False,
                        "v7.0.3": False,
                        "v7.0.2": False,
                        "v7.0.12": False,
                        "v7.0.1": False,
                        "v7.0.0": False,
                        "v6.4.4": False,
                        "v6.4.1": False,
                        "v6.4.0": False,
                        "v6.2.7": False,
                        "v6.2.5": False,
                        "v6.2.3": False,
                        "v6.2.0": False,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
            ],
        },
        "dhcp_ip_overlap": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": False,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": False,
                "v7.0.4": False,
                "v7.0.3": False,
                "v7.0.2": False,
                "v7.0.12": True,
                "v7.0.1": False,
                "v7.0.0": False,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "use-new",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.12": True,
                    },
                },
                {
                    "value": "use-old",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.12": True,
                    },
                },
            ],
        },
        "auto_connect": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "keep_alive": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "save_password": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "ip_pools": {
            "type": "list",
            "elements": "dict",
            "children": {
                "name": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                    "required": True,
                }
            },
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
        },
        "exclusive_routing": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "service_restriction": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "split_tunneling": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "split_tunneling_routing_negate": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                    },
                },
            ],
        },
        "split_tunneling_routing_address": {
            "type": "list",
            "elements": "dict",
            "children": {
                "name": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                    "required": True,
                }
            },
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
        },
        "dns_server1": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "dns_server2": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "dns_suffix": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "wins_server1": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "wins_server2": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "dhcp_ra_giaddr": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": False,
                "v7.2.1": False,
                "v7.2.0": False,
                "v7.0.8": False,
                "v7.0.7": False,
                "v7.0.6": False,
                "v7.0.5": False,
                "v7.0.4": False,
                "v7.0.3": False,
                "v7.0.2": False,
                "v7.0.12": False,
                "v7.0.1": False,
                "v7.0.0": False,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
        },
        "ipv6_tunnel_mode": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "ipv6_pools": {
            "type": "list",
            "elements": "dict",
            "children": {
                "name": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                    "required": True,
                }
            },
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
        },
        "ipv6_exclusive_routing": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "ipv6_service_restriction": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "ipv6_split_tunneling": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "ipv6_split_tunneling_routing_negate": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                    },
                },
            ],
        },
        "ipv6_split_tunneling_routing_address": {
            "type": "list",
            "elements": "dict",
            "children": {
                "name": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                    "required": True,
                }
            },
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
        },
        "ipv6_dns_server1": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "ipv6_dns_server2": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "ipv6_wins_server1": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "ipv6_wins_server2": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "dhcp6_ra_linkaddr": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": False,
                "v7.2.1": False,
                "v7.2.0": False,
                "v7.0.8": False,
                "v7.0.7": False,
                "v7.0.6": False,
                "v7.0.5": False,
                "v7.0.4": False,
                "v7.0.3": False,
                "v7.0.2": False,
                "v7.0.12": False,
                "v7.0.1": False,
                "v7.0.0": False,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
        },
        "client_src_range": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": False,
                "v7.2.1": False,
                "v7.2.0": False,
                "v7.0.8": False,
                "v7.0.7": False,
                "v7.0.6": False,
                "v7.0.5": False,
                "v7.0.4": False,
                "v7.0.3": False,
                "v7.0.2": False,
                "v7.0.12": False,
                "v7.0.1": False,
                "v7.0.0": False,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True, "v7.2.4": True}},
                {"value": "disable", "revisions": {"v7.4.0": True, "v7.2.4": True}},
            ],
        },
        "web_mode": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "landing_page_mode": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": False,
                "v7.2.2": False,
                "v7.2.1": False,
                "v7.2.0": False,
                "v7.0.8": False,
                "v7.0.7": False,
                "v7.0.6": False,
                "v7.0.5": False,
                "v7.0.4": False,
                "v7.0.3": False,
                "v7.0.2": False,
                "v7.0.12": False,
                "v7.0.1": False,
                "v7.0.0": False,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
        "display_bookmark": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "user_bookmark": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "allow_user_access": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "list",
            "options": [
                {
                    "value": "web",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "ftp",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "smb",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "sftp",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "telnet",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "ssh",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "vnc",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "rdp",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "ping",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "citrix",
                    "revisions": {
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "portforward",
                    "revisions": {
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
            "multiple_values": True,
            "elements": "str",
        },
        "user_group_bookmark": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "bookmark_group": {
            "type": "list",
            "elements": "dict",
            "children": {
                "name": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                    "required": True,
                },
                "bookmarks": {
                    "type": "list",
                    "elements": "dict",
                    "children": {
                        "name": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                            "required": True,
                        },
                        "apptype": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "ftp",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "rdp",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "sftp",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": False,
                                        "v6.0.11": False,
                                        "v6.0.0": False,
                                    },
                                },
                                {
                                    "value": "smb",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "ssh",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "telnet",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "vnc",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "web",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "citrix",
                                    "revisions": {
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "portforward",
                                    "revisions": {
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                            ],
                        },
                        "url": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "host": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "folder": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "domain": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": False,
                                "v6.4.0": True,
                                "v6.2.7": False,
                                "v6.2.5": False,
                                "v6.2.3": False,
                                "v6.2.0": False,
                                "v6.0.5": False,
                                "v6.0.11": False,
                                "v6.0.0": False,
                            },
                            "type": "string",
                        },
                        "additional_params": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "description": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "keyboard_layout": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": False,
                                "v6.4.4": False,
                                "v6.4.1": False,
                                "v6.4.0": False,
                                "v6.2.7": False,
                                "v6.2.5": False,
                                "v6.2.3": False,
                                "v6.2.0": False,
                                "v6.0.5": False,
                                "v6.0.11": False,
                                "v6.0.0": False,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "ar-101",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "ar-102",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "ar-102-azerty",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "can-mul",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "cz",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "cz-qwerty",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "cz-pr",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "da",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "nl",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "de",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "de-ch",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "de-ibm",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "en-uk",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "en-uk-ext",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "en-us",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "en-us-dvorak",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "es",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "es-var",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "fi",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "fi-sami",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "fr",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "fr-apple",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": False,
                                        "v7.0.4": False,
                                        "v7.0.3": False,
                                        "v7.0.2": False,
                                        "v7.0.12": True,
                                        "v7.0.1": False,
                                    },
                                },
                                {
                                    "value": "fr-ca",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "fr-ch",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "fr-be",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "hr",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "hu",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "hu-101",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "it",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "it-142",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "ja",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "ko",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "lt",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "lt-ibm",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "lt-std",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "lav-std",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "lav-leg",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "mk",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "mk-std",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "no",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "no-sami",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "pol-214",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "pol-pr",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "pt",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "pt-br",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "pt-br-abnt2",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "ru",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "ru-mne",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "ru-t",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "sl",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "sv",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "sv-sami",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "tuk",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "tur-f",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "tur-q",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "zh-sym-sg-us",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "zh-sym-us",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "zh-tr-hk",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "zh-tr-mo",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "zh-tr-us",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                            ],
                        },
                        "security": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "any",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "rdp",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "nla",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "tls",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                            ],
                        },
                        "send_preconnection_id": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": False,
                                "v6.4.4": False,
                                "v6.4.1": False,
                                "v6.4.0": False,
                                "v6.2.7": False,
                                "v6.2.5": False,
                                "v6.2.3": False,
                                "v6.2.0": False,
                                "v6.0.5": False,
                                "v6.0.11": False,
                                "v6.0.0": False,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "enable",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "disable",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                            ],
                        },
                        "preconnection_id": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "integer",
                        },
                        "preconnection_blob": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "load_balancing_info": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "restricted_admin": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": False,
                                "v6.4.4": False,
                                "v6.4.1": False,
                                "v6.4.0": False,
                                "v6.2.7": False,
                                "v6.2.5": False,
                                "v6.2.3": False,
                                "v6.2.0": False,
                                "v6.0.5": False,
                                "v6.0.11": False,
                                "v6.0.0": False,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "enable",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "disable",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                            ],
                        },
                        "port": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "integer",
                        },
                        "logon_user": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "logon_password": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "color_depth": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": False,
                                "v6.4.4": False,
                                "v6.4.1": False,
                                "v6.4.0": False,
                                "v6.2.7": False,
                                "v6.2.5": False,
                                "v6.2.3": False,
                                "v6.2.0": False,
                                "v6.0.5": False,
                                "v6.0.11": False,
                                "v6.0.0": False,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "32",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "16",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                                {
                                    "value": "8",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                    },
                                },
                            ],
                        },
                        "sso": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "disable",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "static",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "auto",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                            ],
                        },
                        "form_data": {
                            "type": "list",
                            "elements": "dict",
                            "children": {
                                "name": {
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                    "type": "string",
                                    "required": True,
                                },
                                "value": {
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                    "type": "string",
                                },
                            },
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                        },
                        "sso_credential": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "sslvpn-login",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "alternative",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                            ],
                        },
                        "sso_username": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "sso_password": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                        },
                        "sso_credential_sent_once": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "enable",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "disable",
                                    "revisions": {
                                        "v7.4.0": True,
                                        "v7.2.4": True,
                                        "v7.2.2": True,
                                        "v7.2.1": True,
                                        "v7.2.0": True,
                                        "v7.0.8": True,
                                        "v7.0.7": True,
                                        "v7.0.6": True,
                                        "v7.0.5": True,
                                        "v7.0.4": True,
                                        "v7.0.3": True,
                                        "v7.0.2": True,
                                        "v7.0.12": True,
                                        "v7.0.1": True,
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                            ],
                        },
                        "width": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": False,
                                "v7.0.2": False,
                                "v7.0.12": True,
                                "v7.0.1": False,
                                "v7.0.0": False,
                                "v6.4.4": False,
                                "v6.4.1": False,
                                "v6.4.0": False,
                                "v6.2.7": False,
                                "v6.2.5": False,
                                "v6.2.3": False,
                                "v6.2.0": False,
                                "v6.0.5": False,
                                "v6.0.11": False,
                                "v6.0.0": False,
                            },
                            "type": "integer",
                        },
                        "height": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": False,
                                "v7.0.2": False,
                                "v7.0.12": True,
                                "v7.0.1": False,
                                "v7.0.0": False,
                                "v6.4.4": False,
                                "v6.4.1": False,
                                "v6.4.0": False,
                                "v6.2.7": False,
                                "v6.2.5": False,
                                "v6.2.3": False,
                                "v6.2.0": False,
                                "v6.0.5": False,
                                "v6.0.11": False,
                                "v6.0.0": False,
                            },
                            "type": "integer",
                        },
                        "vnc_keyboard_layout": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": False,
                                "v7.2.1": False,
                                "v7.2.0": False,
                                "v7.0.8": False,
                                "v7.0.7": False,
                                "v7.0.6": False,
                                "v7.0.5": False,
                                "v7.0.4": False,
                                "v7.0.3": False,
                                "v7.0.2": False,
                                "v7.0.12": False,
                                "v7.0.1": False,
                                "v7.0.0": False,
                                "v6.4.4": False,
                                "v6.4.1": False,
                                "v6.4.0": False,
                                "v6.2.7": False,
                                "v6.2.5": False,
                                "v6.2.3": False,
                                "v6.2.0": False,
                                "v6.0.5": False,
                                "v6.0.11": False,
                                "v6.0.0": False,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "default",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "da",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "nl",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "en-uk",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "en-uk-ext",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "fi",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "fr",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "fr-be",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "fr-ca-mul",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "de",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "de-ch",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "it",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "it-142",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "pt",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "pt-br-abnt2",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "no",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "gd",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "es",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "sv",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                                {
                                    "value": "us-intl",
                                    "revisions": {"v7.4.0": True, "v7.2.4": True},
                                },
                            ],
                        },
                        "listening_port": {
                            "revisions": {
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "integer",
                        },
                        "remote_port": {
                            "revisions": {
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "integer",
                        },
                        "show_status_window": {
                            "revisions": {
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "enable",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "disable",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                            ],
                        },
                        "server_layout": {
                            "revisions": {
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                            "options": [
                                {
                                    "value": "de-de-qwertz",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "en-gb-qwerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "en-us-qwerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "es-es-qwerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "fr-ca-qwerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": False,
                                        "v6.0.11": False,
                                        "v6.0.0": False,
                                    },
                                },
                                {
                                    "value": "fr-fr-azerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "fr-ch-qwertz",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "it-it-qwerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "ja-jp-qwerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "pt-br-qwerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "sv-se-qwerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "tr-tr-qwerty",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                                {
                                    "value": "failsafe",
                                    "revisions": {
                                        "v7.0.0": True,
                                        "v6.4.4": True,
                                        "v6.4.1": True,
                                        "v6.4.0": True,
                                        "v6.2.7": True,
                                        "v6.2.5": True,
                                        "v6.2.3": True,
                                        "v6.2.0": True,
                                        "v6.0.5": True,
                                        "v6.0.11": True,
                                        "v6.0.0": True,
                                    },
                                },
                            ],
                        },
                    },
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            },
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
        },
        "display_connection_tools": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "display_history": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "display_status": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "rewrite_ip_uri_ui": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                    },
                },
            ],
        },
        "heading": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "redir_url": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "theme": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "jade",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": False,
                        "v6.4.1": False,
                        "v6.4.0": False,
                        "v6.2.7": False,
                        "v6.2.5": False,
                        "v6.2.3": False,
                        "v6.2.0": False,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "neutrino",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "mariner",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "graphite",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": False,
                        "v6.4.1": False,
                        "v6.4.0": False,
                        "v6.2.7": False,
                        "v6.2.5": False,
                        "v6.2.3": False,
                        "v6.2.0": False,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "melongene",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "jet-stream",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": False,
                        "v7.2.2": False,
                        "v7.2.1": False,
                        "v7.2.0": False,
                        "v7.0.8": False,
                        "v7.0.7": False,
                        "v7.0.6": False,
                        "v7.0.5": False,
                        "v7.0.4": False,
                        "v7.0.3": False,
                        "v7.0.2": False,
                        "v7.0.12": False,
                        "v7.0.1": False,
                        "v7.0.0": False,
                        "v6.4.4": False,
                        "v6.4.1": False,
                        "v6.4.0": False,
                        "v6.2.7": False,
                        "v6.2.5": False,
                        "v6.2.3": False,
                        "v6.2.0": False,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "security-fabric",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": False,
                        "v7.2.2": False,
                        "v7.2.1": False,
                        "v7.2.0": False,
                        "v7.0.8": False,
                        "v7.0.7": False,
                        "v7.0.6": False,
                        "v7.0.5": False,
                        "v7.0.4": False,
                        "v7.0.3": False,
                        "v7.0.2": False,
                        "v7.0.12": False,
                        "v7.0.1": False,
                        "v7.0.0": False,
                        "v6.4.4": False,
                        "v6.4.1": False,
                        "v6.4.0": False,
                        "v6.2.7": False,
                        "v6.2.5": False,
                        "v6.2.3": False,
                        "v6.2.0": False,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "dark-matter",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": False,
                        "v6.4.1": False,
                        "v6.4.0": False,
                        "v6.2.7": False,
                        "v6.2.5": False,
                        "v6.2.3": False,
                        "v6.2.0": False,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "onyx",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": False,
                        "v6.4.1": False,
                        "v6.4.0": False,
                        "v6.2.7": False,
                        "v6.2.5": False,
                        "v6.2.3": False,
                        "v6.2.0": False,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "eclipse",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": False,
                        "v6.4.1": False,
                        "v6.4.0": False,
                        "v6.2.7": False,
                        "v6.2.5": False,
                        "v6.2.3": False,
                        "v6.2.0": False,
                        "v6.0.5": False,
                        "v6.0.11": False,
                        "v6.0.0": False,
                    },
                },
                {
                    "value": "blue",
                    "revisions": {
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "green",
                    "revisions": {
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "red",
                    "revisions": {"v6.0.5": True, "v6.0.11": True, "v6.0.0": True},
                },
            ],
        },
        "custom_lang": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "smb_ntlmv1_auth": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "smbv1": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "smb_min_version": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "smbv1",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
                {
                    "value": "smbv2",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
                {
                    "value": "smbv3",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
            ],
        },
        "smb_max_version": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "smbv1",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
                {
                    "value": "smbv2",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
                {
                    "value": "smbv3",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
            ],
        },
        "use_sdwan": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": True,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.2.7": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.2.7": True,
                    },
                },
            ],
        },
        "prefer_ipv6_dns": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                    },
                },
            ],
        },
        "clipboard": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": False,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                    },
                },
            ],
        },
        "default_window_width": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": False,
                "v7.0.4": False,
                "v7.0.3": False,
                "v7.0.2": False,
                "v7.0.12": True,
                "v7.0.1": False,
                "v7.0.0": False,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "integer",
        },
        "default_window_height": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": False,
                "v7.0.4": False,
                "v7.0.3": False,
                "v7.0.2": False,
                "v7.0.12": True,
                "v7.0.1": False,
                "v7.0.0": False,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "integer",
        },
        "host_check": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "none",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "av",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "fw",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "av-fw",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "custom",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "host_check_interval": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "integer",
        },
        "host_check_policy": {
            "type": "list",
            "elements": "dict",
            "children": {
                "name": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                    "required": True,
                }
            },
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
        },
        "limit_user_logins": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "mac_addr_check": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "mac_addr_action": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "allow",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "deny",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "mac_addr_check_rule": {
            "type": "list",
            "elements": "dict",
            "children": {
                "name": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                    "required": True,
                },
                "mac_addr_mask": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "integer",
                },
                "mac_addr_list": {
                    "type": "list",
                    "elements": "dict",
                    "children": {
                        "addr": {
                            "revisions": {
                                "v7.4.0": True,
                                "v7.2.4": True,
                                "v7.2.2": True,
                                "v7.2.1": True,
                                "v7.2.0": True,
                                "v7.0.8": True,
                                "v7.0.7": True,
                                "v7.0.6": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.12": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                            "type": "string",
                            "required": True,
                        }
                    },
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            },
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
        },
        "os_check": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "forticlient_download": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "forticlient_download_method": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "direct",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "ssl-vpn",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "customize_forticlient_download_url": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "windows_forticlient_download_url": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "macos_forticlient_download_url": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
        },
        "skip_check_for_unsupported_os": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "skip_check_for_browser": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
            ],
        },
        "hide_sso_credential": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                },
            ],
        },
        "split_dns": {
            "type": "list",
            "elements": "dict",
            "children": {
                "id": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "integer",
                    "required": True,
                },
                "domains": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                },
                "dns_server1": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                },
                "dns_server2": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                },
                "ipv6_dns_server1": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                },
                "ipv6_dns_server2": {
                    "revisions": {
                        "v7.4.0": True,
                        "v7.2.4": True,
                        "v7.2.2": True,
                        "v7.2.1": True,
                        "v7.2.0": True,
                        "v7.0.8": True,
                        "v7.0.7": True,
                        "v7.0.6": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.12": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                },
            },
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": True,
                "v7.2.2": True,
                "v7.2.1": True,
                "v7.2.0": True,
                "v7.0.8": True,
                "v7.0.7": True,
                "v7.0.6": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.12": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
        },
        "landing_page": {
            "revisions": {
                "v7.4.0": True,
                "v7.2.4": False,
                "v7.2.2": False,
                "v7.2.1": False,
                "v7.2.0": False,
                "v7.0.8": False,
                "v7.0.7": False,
                "v7.0.6": False,
                "v7.0.5": False,
                "v7.0.4": False,
                "v7.0.3": False,
                "v7.0.2": False,
                "v7.0.12": False,
                "v7.0.1": False,
                "v7.0.0": False,
                "v6.4.4": False,
                "v6.4.1": False,
                "v6.4.0": False,
                "v6.2.7": False,
                "v6.2.5": False,
                "v6.2.3": False,
                "v6.2.0": False,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "dict",
            "children": {
                "url": {"revisions": {"v7.4.0": True}, "type": "string"},
                "logout_url": {"revisions": {"v7.4.0": True}, "type": "string"},
                "sso": {
                    "revisions": {"v7.4.0": True},
                    "type": "string",
                    "options": [
                        {"value": "disable", "revisions": {"v7.4.0": True}},
                        {"value": "static", "revisions": {"v7.4.0": True}},
                        {"value": "auto", "revisions": {"v7.4.0": True}},
                    ],
                },
                "form_data": {
                    "type": "list",
                    "elements": "dict",
                    "children": {
                        "name": {
                            "revisions": {"v7.4.0": True},
                            "type": "string",
                            "required": True,
                        },
                        "value": {"revisions": {"v7.4.0": True}, "type": "string"},
                    },
                    "revisions": {"v7.4.0": True},
                },
                "sso_credential": {
                    "revisions": {"v7.4.0": True},
                    "type": "string",
                    "options": [
                        {"value": "sslvpn-login", "revisions": {"v7.4.0": True}},
                        {"value": "alternative", "revisions": {"v7.4.0": True}},
                    ],
                },
                "sso_username": {"revisions": {"v7.4.0": True}, "type": "string"},
                "sso_password": {"revisions": {"v7.4.0": True}, "type": "string"},
            },
        },
        "os_check_list": {
            "type": "list",
            "elements": "dict",
            "children": {
                "name": {
                    "revisions": {
                        "v7.2.0": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                    "required": True,
                },
                "action": {
                    "revisions": {
                        "v7.2.0": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                    "options": [
                        {
                            "value": "deny",
                            "revisions": {
                                "v7.2.0": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                        },
                        {
                            "value": "allow",
                            "revisions": {
                                "v7.2.0": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                        },
                        {
                            "value": "check-up-to-date",
                            "revisions": {
                                "v7.2.0": True,
                                "v7.0.5": True,
                                "v7.0.4": True,
                                "v7.0.3": True,
                                "v7.0.2": True,
                                "v7.0.1": True,
                                "v7.0.0": True,
                                "v6.4.4": True,
                                "v6.4.1": True,
                                "v6.4.0": True,
                                "v6.2.7": True,
                                "v6.2.5": True,
                                "v6.2.3": True,
                                "v6.2.0": True,
                                "v6.0.5": True,
                                "v6.0.11": True,
                                "v6.0.0": True,
                            },
                        },
                    ],
                },
                "tolerance": {
                    "revisions": {
                        "v7.2.0": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "integer",
                },
                "latest_patch_level": {
                    "revisions": {
                        "v7.2.0": True,
                        "v7.0.5": True,
                        "v7.0.4": True,
                        "v7.0.3": True,
                        "v7.0.2": True,
                        "v7.0.1": True,
                        "v7.0.0": True,
                        "v6.4.4": True,
                        "v6.4.1": True,
                        "v6.4.0": True,
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                        "v6.0.5": True,
                        "v6.0.11": True,
                        "v6.0.0": True,
                    },
                    "type": "string",
                },
            },
            "revisions": {
                "v7.2.0": True,
                "v7.0.5": True,
                "v7.0.4": True,
                "v7.0.3": True,
                "v7.0.2": True,
                "v7.0.1": True,
                "v7.0.0": True,
                "v6.4.4": True,
                "v6.4.1": True,
                "v6.4.0": True,
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": True,
                "v6.0.11": True,
                "v6.0.0": True,
            },
        },
        "transform_backward_slashes": {
            "revisions": {
                "v6.2.7": True,
                "v6.2.5": True,
                "v6.2.3": True,
                "v6.2.0": True,
                "v6.0.5": False,
                "v6.0.11": False,
                "v6.0.0": False,
            },
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
                {
                    "value": "disable",
                    "revisions": {
                        "v6.2.7": True,
                        "v6.2.5": True,
                        "v6.2.3": True,
                        "v6.2.0": True,
                    },
                },
            ],
        },
        "skip_check_for_unsupported_browser": {
            "revisions": {"v6.0.5": True, "v6.0.11": True, "v6.0.0": True},
            "type": "string",
            "options": [
                {
                    "value": "enable",
                    "revisions": {"v6.0.5": True, "v6.0.11": True, "v6.0.0": True},
                },
                {
                    "value": "disable",
                    "revisions": {"v6.0.5": True, "v6.0.11": True, "v6.0.0": True},
                },
            ],
        },
    },
    "revisions": {
        "v7.4.0": True,
        "v7.2.4": True,
        "v7.2.2": True,
        "v7.2.1": True,
        "v7.2.0": True,
        "v7.0.8": True,
        "v7.0.7": True,
        "v7.0.6": True,
        "v7.0.5": True,
        "v7.0.4": True,
        "v7.0.3": True,
        "v7.0.2": True,
        "v7.0.12": True,
        "v7.0.1": True,
        "v7.0.0": True,
        "v6.4.4": True,
        "v6.4.1": True,
        "v6.4.0": True,
        "v6.2.7": True,
        "v6.2.5": True,
        "v6.2.3": True,
        "v6.2.0": True,
        "v6.0.5": True,
        "v6.0.11": True,
        "v6.0.0": True,
    },
}


def main():
    module_spec = schema_to_module_spec(versioned_schema)
    mkeyname = "name"
    fields = {
        "access_token": {"required": False, "type": "str", "no_log": True},
        "enable_log": {"required": False, "type": "bool", "default": False},
        "vdom": {"required": False, "type": "str", "default": "root"},
        "member_path": {"required": False, "type": "str"},
        "member_state": {
            "type": "str",
            "required": False,
            "choices": ["present", "absent"],
        },
        "state": {"required": True, "type": "str", "choices": ["present", "absent"]},
        "vpn_ssl_web_portal": {
            "required": False,
            "type": "dict",
            "default": None,
            "options": {},
        },
    }
    for attribute_name in module_spec["options"]:
        fields["vpn_ssl_web_portal"]["options"][attribute_name] = module_spec[
            "options"
        ][attribute_name]
        if mkeyname and mkeyname == attribute_name:
            fields["vpn_ssl_web_portal"]["options"][attribute_name]["required"] = True

    module = AnsibleModule(argument_spec=fields, supports_check_mode=True)
    check_legacy_fortiosapi(module)

    is_error = False
    has_changed = False
    result = None
    diff = None

    versions_check_result = None
    if module._socket_path:
        connection = Connection(module._socket_path)
        if "access_token" in module.params:
            connection.set_option("access_token", module.params["access_token"])

        if "enable_log" in module.params:
            connection.set_option("enable_log", module.params["enable_log"])
        else:
            connection.set_option("enable_log", False)
        fos = FortiOSHandler(connection, module, mkeyname)
        versions_check_result = check_schema_versioning(
            fos, versioned_schema, "vpn_ssl_web_portal"
        )

        is_error, has_changed, result, diff = fortios_vpn_ssl_web(
            module.params, fos, module.check_mode
        )

    else:
        module.fail_json(**FAIL_SOCKET_MSG)

    if versions_check_result and versions_check_result["matched"] is False:
        module.warn(
            "Ansible has detected version mismatch between FortOS system and your playbook, see more details by specifying option -vvv"
        )

    if not is_error:
        if versions_check_result and versions_check_result["matched"] is False:
            module.exit_json(
                changed=has_changed,
                version_check_warning=versions_check_result,
                meta=result,
                diff=diff,
            )
        else:
            module.exit_json(changed=has_changed, meta=result, diff=diff)
    else:
        if versions_check_result and versions_check_result["matched"] is False:
            module.fail_json(
                msg="Error in repo",
                version_check_warning=versions_check_result,
                meta=result,
            )
        else:
            module.fail_json(msg="Error in repo", meta=result)


if __name__ == "__main__":
    main()
